% N-class classification with confusion matrix
% Train a neural classifier to detect thyroid malfunctioning 
% 21 features describing patient attributes
% Three classes corresponding to: normal, hyperthyroidism, hypothyroidism
% Load the dataset with the command
% [x,t] = thyroid_dataset;
% Divide in training and test sets
% Try to optimize the parameters to minimize test error rate
% Analyze the results using a confusion matrix


close all
clear variables


[x,t] = thyroid_dataset;
size(x)
size(t)

P = x;
T = t;

% Divide into training and test DB
[trainInd,valInd,testInd] = dividerand(size(P,2),0.7,0,0.3);

P_train=P(:,trainInd);
T_train=T(:,trainInd);
P_test=P(:,testInd);
T_test=T(:,testInd);


% create a neural network
neuronsXLayer = 20; % number of neurons per layer
neuronTopology{1} = 'tansig';


net = feedforwardnet(neuronsXLayer);

% training and testing data
net.divideParam.trainRatio = 1; 
net.divideParam.testRatio  = 0; 
net.divideParam.valRatio   = 0;

net.trainParam.epochs = 100;
net.trainParam.goal = 0.01;
for iL = 1: size(neuronsXLayer,2)
	net.layers{iL}.transferFcn = neuronTopology{iL}; 
end


% train a neural network
[net,tr,Y,E] = train(net,P,T);


% Training and testing data
trainResult = net(P_train);
[maxValueTrain,trainResult] = max(trainResult);
[~,expectedTrainResult]=max(T_train);

testResultMultiNeuron = net(P_test);
[maxValueTest,testResult] = max(testResultMultiNeuron);
[~,expectedTestResult]=max(T_test);


% %plot
% figure,
% plot(expectedTrainResult)
% hold on
% plot(trainResult, '--r')
% title('training results')
% 
% figure,
% plot(expectedTestResult)
% hold on
% plot(testResult, '--r')
% title('testing results')

trainErr = expectedTrainResult ~= trainResult;
testErr = expectedTestResult ~= testResult;


% figure,
% plot(trainErr)
% title('training error')
% 
% figure,
% plot(testErr)
% title('testing error')



%Results
numTrainErr = size(find(trainErr > 0),2);
percTrainErr = (numTrainErr/size(T_train,2))*100;


numTestErr = size(find(testErr > 0),2);
percTestErr = (numTestErr/size(T_test,2))*100;


totalErr = [trainErr, testErr];
numTotalErr = size(find(totalErr > 0),2);
percTotalErr = (numTotalErr/size(T,2))*100;

fprintf('\nRESULTS \n')
fprintf('TRAINING\n')
fprintf('n. of el. = %i \n', size(trainErr,2))
fprintf('n. of errors = %i \n', numTrainErr)
fprintf('Error rate = %6.2f %%\n', percTrainErr)
fprintf('Classification accuracy = %6.2f %%\n', 100-percTrainErr)
fprintf('TESTING\n')
fprintf('n. of el. = %i \n', size(testErr,2))
fprintf('n. of errors = %i \n', numTestErr)
fprintf('Error rate = %6.2f %%\n', percTestErr)
fprintf('Classification accuracy = %6.2f %%\n', 100-percTestErr)
fprintf('TOTAL\n')
fprintf('n. of el. = %i \n', size(totalErr,2))
fprintf('n. of errors = %i \n', numTotalErr)
fprintf('Error rate = %6.2f %%\n', percTotalErr)
fprintf('Classification accuracy = %6.2f %%\n', 100-percTotalErr)


% Confusion matrix
cm = confusionmat(expectedTestResult,testResult);
fprintf('Confusion matrix \n\n')
cm

plotconfusion(T_test,testResultMultiNeuron);

