% Two-class classification with confusion matrix
% Train a neural classifier to identify the gender of crabs from physical dimensions of the crab:
% Six physical characterstics of a crab are considered: species, frontallip, rearwidth, length, width and depth.
% 2 classes (male, female)
% Load the dataset using the commands:
% [x,t] = crab_dataset;
% P = x;
% T = t(1,:);
% Divide in training and test sets
% Try to optimize the parameters to minimize test error rate
% Analyze the results using a confusion matrix


close all
clear variables

[x,t] = crab_dataset;
size(x)
size(t)

P = x;
T = t(1,:);
numClasses=2;

[trainInd,valInd,testInd] = dividerand(size(P,2),0.7,0,0.3);

P_train=P(:,trainInd);
T_train=T(:,trainInd);
P_test=P(:,testInd);
T_test=T(:,testInd);


% create a neural network
neuronsXLayer = 30; % number of neurons per layer
neuronTopology{1} = 'tansig';


net = feedforwardnet(neuronsXLayer);

% training and testing data
net.divideParam.trainRatio = 1; 
net.divideParam.testRatio  = 0; 
net.divideParam.valRatio   = 0;

net.trainParam.epochs = 100;
net.trainParam.goal = 0.01;
for iL = 1: size(neuronsXLayer,2)
	net.layers{iL}.transferFcn = neuronTopology{iL}; 
end


% train a neural network
[net,tr,Y,E] = train(net,P_train,T_train);


% Training and testing data
trainResult = net(P_train);
trainResult = round(trainResult);
trainResult(trainResult<0)=0;
trainResult(trainResult>numClasses-1)=numClasses-1;


testResult = net(P_test);
testResult=round(testResult);
testResult(testResult<0)=0;
testResult(testResult>numClasses-1)=numClasses-1;

trainErr = T_train ~= trainResult;
testErr = T_test ~= testResult;


figure,
plot(trainErr)
title('training error')

figure,
plot(testErr)
title('testing error')


%Results
numTrainErr = size(find(trainErr > 0),2);
percTrainErr = (numTrainErr/size(T_train,2))*100;


numTestErr = size(find(testErr > 0),2);
percTestErr = (numTestErr/size(T_test,2))*100;


totalErr = [trainErr, testErr];
numTotalErr = size(find(totalErr > 0),2);
percTotalErr = (numTotalErr/size(T,2))*100;

fprintf('\nRESULTS \n')
fprintf('TRAINING\n')
fprintf('n. of el. = %i \n', size(trainErr,2))
fprintf('n. of errors = %i \n', numTrainErr)
fprintf('Error rate = %6.2f %%\n', percTrainErr)
fprintf('Classification accuracy = %6.2f %%\n', 100-percTrainErr)
fprintf('TESTING\n')
fprintf('n. of el. = %i \n', size(testErr,2))
fprintf('n. of errors = %i \n', numTestErr)
fprintf('Error rate = %6.2f %%\n', percTestErr)
fprintf('Classification accuracy = %6.2f %%\n', 100-percTestErr)
fprintf('TOTAL\n')
fprintf('n. of el. = %i \n', size(totalErr,2))
fprintf('n. of errors = %i \n', numTotalErr)
fprintf('Error rate = %6.2f %%\n', percTotalErr)
fprintf('Classification accuracy = %6.2f %%\n', 100-percTotalErr)

% Confusion matrix
cm = confusionmat(T_test,testResult);
fprintf('Confusion matrix \n\n')
cm

figure
plotconfusion(T_test,testResult);

