#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "random.h"
#include "greedy.h"
#include "localsearch.h"
#include "scattersearch.h"
#include "pathrelinking.h"
#include "randomsolution.h"


void parse_command_line (int argc, char *argv[], char *data_file, char *algo, double *ptauMax,
                         long *pseed, int *pnb, int *pnd);


int main (int argc, char *argv[])
{
  char data_file[NAME_LENGTH];
  data_t I;
  solution_t x;
  char algo[NAME_LENGTH];
  double tauMax;
  int nb, nd;
  long seed;
  time_t start, end;
  double tempo;


  parse_command_line(argc,argv,data_file,algo,&tauMax,&seed,&nb,&nd);

  load_data(data_file,&I);

  create_solution(I.n,&x);

  start = clock();
  if (strcmp(algo,"-ss") == 0)
    scatter_search(&I,&x,"-fb",tauMax,nb,nd,&seed);
  else if (strcmp(algo,"-pr") == 0)
    path_relinking(&I,&x,"-fb",tauMax,nb,&seed);
  else if (strcmp(algo,"-rr") == 0)
    random_restart(&I,&x,"-fb",tauMax,&seed);
  end = clock();
  tempo = (double) (end - start) / CLOCKS_PER_SEC;

  printf("%s ",data_file);
  printf("%10.6f ",tempo);
  print_sorted_solution(&x,I.n);
  printf("\n");

  destroy_solution(&x);
  destroy_data(&I);

  return EXIT_SUCCESS;
}


void print_usage (char *command)
{
  fprintf(stderr,"Use: %s [datafile] [-ss time nb nd seed]|[-pr time nb seed]\n",command);
  fprintf(stderr,"datafile:     name and path of data file\n");
  fprintf(stderr,"-ss:          scatter search\n");
  fprintf(stderr,"     time:    time limit\n");
  fprintf(stderr,"     seed:    seed of the pseudorandom number generator\n");
  fprintf(stderr,"     nb:      number of solutions in the best subset\n");
  fprintf(stderr,"     nd:      number of solutions in the diverse subset\n");
  fprintf(stderr,"-pr:          path relinking\n");
  fprintf(stderr,"     time:    time limit\n");
  fprintf(stderr,"     seed:    seed of the pseudorandom number generator\n");
  fprintf(stderr,"     nb:      number of solutions in the best subset\n");
  fprintf(stderr,"-rr:          random restart (for comparison)\n");
  fprintf(stderr,"     time:    time limit\n");
  fprintf(stderr,"     seed:    seed of the pseudorandom number generator\n");
}


void parse_command_line (int argc, char *argv[], char *data_file, char *algo, double *ptauMax,
                         long *pseed, int *pnb, int *pnd)
{
  if ( (argc != 5) && (argc != 6) && (argc != 7) )
  {
    print_usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  strcpy(data_file,argv[1]);
  strcpy(algo,argv[2]);
  if (strcmp(algo,"-ss") == 0)
  {
    sscanf(argv[3],"%lf",ptauMax);
    sscanf(argv[4],"%ld",pseed);
    sscanf(argv[5],"%d",pnb);
    sscanf(argv[6],"%d",pnd);
  }
  else if (strcmp(algo,"-pr") == 0)
  {
    sscanf(argv[3],"%lf",ptauMax);
    sscanf(argv[4],"%ld",pseed);
    sscanf(argv[5],"%d",pnb);
  }
  else if (strcmp(algo,"-rr") == 0)
  {
    sscanf(argv[3],"%lf",ptauMax);
    sscanf(argv[4],"%ld",pseed);
  }
  else
  {
    print_usage(argv[0]);
    exit(EXIT_FAILURE);
  }
}
