#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "random.h"
#include "greedy.h"
#include "localsearch.h"
#include "vns.h"
#include "tabusearch.h"


void parse_command_line (int argc, char *argv[], char *data_file, char *exchange_algo, int *pniter,
                         long *pseed, int *ps_min, int *pdelta_s, int *ps_max, int *pl_in, int *pl_out);


int main (int argc, char *argv[])
{
  char data_file[NAME_LENGTH];
  data_t I;
  solution_t x;
  char exchange_algo[NAME_LENGTH];
  long seed;
  time_t start, end;
  double tempo;
  int niter;
  int s_min, delta_s, s_max;
  int l_in, l_out;


  parse_command_line(argc,argv,data_file,exchange_algo,&niter,&seed,&s_min,&delta_s,&s_max,&l_in,&l_out);

  load_data(data_file,&I);

  create_solution(I.n,&x);
  greedy_farthest(&I,&x);

  start = clock();
  if (strcmp(exchange_algo,"-vns") == 0)
    variable_neighborhood_search(&I,&x,"-fb",niter,s_min,delta_s,s_max,&seed);
  else if (strcmp(exchange_algo,"-ts") == 0)
    tabu_search(&I,&x,"-fb",niter,l_in,l_out);
  end = clock();
  tempo = (double) (end - start) / CLOCKS_PER_SEC;

  printf("%s ",data_file);
  printf("%10.6f ",tempo);
  printf("%8d ",niter);
  print_sorted_solution(&x,I.n);
  printf("\n");

  destroy_solution(&x);
  destroy_data(&I);

  return EXIT_SUCCESS;
}


void print_usage (char *command)
{
  fprintf(stderr,"Use: %s [datafile] [-vns niter s_min delta_s s_max seed]|[-ts niter l_min l_max]\n",command);
  fprintf(stderr,"datafile:      name and path of data file\n");
  fprintf(stderr,"-vns:          variable neighbourhood search\n");
  fprintf(stderr,"     niter:    total number of neighbourhood explorations\n");
  fprintf(stderr,"     s_min:    minimum number of swaps in the shaking phase\n");
  fprintf(stderr,"     delta_s:  increase in the number of swaps\n");
  fprintf(stderr,"     s_max:    maximum number of swaps in the shaking phase\n");
  fprintf(stderr,"     seed:     seed of the pseudorandom number generator\n");
  fprintf(stderr,"-ts:           tabu search\n");
  fprintf(stderr,"     n_iter:   total number of neighbourhood explorations\n");
  fprintf(stderr,"     l_in:     tabu tenure for insertions\n");
  fprintf(stderr,"     l_out:    tabu tenure for removals\n");
}


void parse_command_line (int argc, char *argv[], char *data_file, char *exchange_algo, int *pniter,
                         long *pseed, int *ps_min, int *pdelta_s, int *ps_max, int *pl_in, int *pl_out)
{
  if ( (argc != 6) && (argc != 8) )
  {
    print_usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  strcpy(data_file,argv[1]);
  strcpy(exchange_algo,argv[2]);
  if (strcmp(exchange_algo,"-vns") == 0)
  {
    sscanf(argv[3],"%d",pniter);
    sscanf(argv[4],"%d",ps_min);
    sscanf(argv[5],"%d",pdelta_s);
    sscanf(argv[6],"%d",ps_max);
    sscanf(argv[7],"%ld",pseed);
  }
  else if (strcmp(exchange_algo,"-ts") == 0)
  {
    sscanf(argv[3],"%d",pniter);
    sscanf(argv[4],"%d",pl_in);
    sscanf(argv[5],"%d",pl_out);
  }
  else
  {
    print_usage(argv[0]);
    exit(EXIT_FAILURE);
  }
}
