#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "greedy.h"
#include "localsearch.h"
#include "random.h"


void parse_command_line (int argc, char *argv[], char *data_file, char *init_algo, char *visit_strategy, long *pseed);

void generate_random_solution (data_t *pI, solution_t *px, long *pseed);


int main (int argc, char *argv[])
{
  char data_file[NAME_LENGTH];
  data_t I;
  solution_t x;
  char init_algo[NAME_LENGTH];
  char visit_strategy[NAME_LENGTH];
  long seed;
  time_t start, end;
  double tempo;
  int niter;


  parse_command_line(argc,argv,data_file,init_algo,visit_strategy,&seed);

  load_data(data_file,&I);

  create_solution(I.n,&x);

  start = clock();
  if (strcmp(init_algo, "-gf") == 0)
    greedy_farthest(&I, &x);
  else if (strcmp(init_algo, "-ga") == 0)
    greedy_tryall(&I, &x);
  else if (strcmp(init_algo, "-r") == 0)
    generate_random_solution(&I, &x, &seed);
  steepest_ascent(&I,&x,visit_strategy,&niter);
  end = clock();
  tempo = (double) (end - start) / CLOCKS_PER_SEC;

  printf("%s ",data_file);
  printf("%10.6f ",tempo);
  printf("%8d ",niter);
  print_sorted_solution(&x,I.n);
  printf("\n");

  destroy_solution(&x);
  destroy_data(&I);

  return EXIT_SUCCESS;
}


void print_usage (char *command)
{
  fprintf(stderr,"Use: %s [datafile] [-gf|-ga|-r seed] [-fb|gb]\n",command);
  fprintf(stderr,"datafile:      name and path of data file\n");
  fprintf(stderr,"-gf:           greedy heuristic starting from point with the largest distance sum\n");
  fprintf(stderr,"-ga:           greedy heuristic starting from each point\n");
  fprintf(stderr,"-r seed:       random initial solution (seed = negative integer)\n");
  fprintf(stderr,"-fb:           first-best neighbourhood exploration strategy\n");
  fprintf(stderr,"-gb:           global-best neighbourhood exploration strategy\n");
}


void parse_command_line (int argc, char *argv[], char *data_file, char *init_algo, char *visit_strategy, long *pseed)
{
  if ( (argc != 4) && (argc != 5) )
  {
    print_usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  strcpy(data_file,argv[1]);
  strcpy(init_algo,argv[2]);
  if (strcmp(init_algo,"-r") != 0)
    strcpy(visit_strategy,argv[3]);
  else
  {
    sscanf(argv[4],"%ld",pseed);
    strcpy(visit_strategy,argv[4]);
  }
}


void generate_random_solution (data_t* pI, solution_t* px, long* pseed)
{
  int* index;
  int i, num;

  index = int_alloc(pI->n+1);
  for (i = 1; i <= pI->n; i++)
    index[i] = i;

  num = 0;
  while (num < pI->k)
  {
    i = rand_int(1,pI->n-num,pseed);
    add_point(get_point(index[i],pI),px,pI);
    index[i] = index[pI->n-num];
    num++;
  }

  free(index);
}
