#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "ant_system.h"
#include "grasp.h"


void parse_command_line (int argc, char *argv[], char *data_file, char *algo, int *piterations, double *pmu, double *pq, double *prho, long *pseed);


int main (int argc, char *argv[])
{
  char data_file[NAME_LENGTH];
  data_t I;
  solution_t x;
  char algo[NAME_LENGTH];
  time_t inizio, fine;
  double tempo;
  long seed;
  int iterations;
  double mu;
  double q, rho;
  int arg;


  parse_command_line (argc,argv,data_file,algo,&iterations,&mu,&q,&rho,&seed);

  load_data(data_file,&I);
  /*print_data(&I);*/

  create_solution(I.n,&x);

  inizio = clock();
  if (strcmp(algo,"-grasp") == 0)
    grasp(&I,&x,iterations,mu,&seed);
  else if (strcmp(algo,"-as") == 0)
    ant_system(&I,&x,iterations,q,rho,&seed);

  fine = clock();
  tempo = (double) (fine - inizio) / CLOCKS_PER_SEC;

  printf("%s ",data_file);
  for (arg = 2; arg < argc; arg++)
    printf("%s ",argv[arg]);
  printf("%10.6f ",tempo);
  print_sorted_solution(&x,I.n);
  printf("\n");

  destroy_solution(&x);
  destroy_data(&I);

  return EXIT_SUCCESS;
}


void print_usage (char *command)
{
  fprintf(stderr,"Use: %s [datafile] [-grasp|-as] [parameters] [seed]\n",command);
  fprintf(stderr,"datafile:        name and path of data file\n");
  fprintf(stderr,"-grasp iter mu:  GRASP with iter iterations and RCL parameter mu\n");
  fprintf(stderr,"-as iter q rho:  Ant System with iter iterations, randomness q and oblivion rho\n");
  fprintf(stderr,"seed:            seed of the pseudorandom number generator (negative integer)\n");
}


void parse_command_line (int argc, char *argv[], char *data_file, char *algo, int *piterations, double *pmu, double *pq, double *prho, long *pseed)
{
  if ( (argc != 6) && (argc != 7) )
  {
    print_usage(argv[0]);
    exit(EXIT_FAILURE);
  }

  *piterations = 0;
  *pmu = 0.0;
  *pq = 0.0;
  *prho = 1.0;
  *pseed = -1;

  strcpy(data_file,argv[1]);
  strcpy(algo,argv[2]);
  if (strcmp(algo, "-grasp") == 0)
  {
    sscanf(argv[3],"%d",piterations);
    sscanf(argv[4],"%lf",pmu);
    sscanf(argv[5],"%ld",pseed);
  }
  else if (strcmp(algo,"-as") == 0)
  {
    sscanf(argv[3],"%d",piterations);
    sscanf(argv[4],"%lf",pq);
    sscanf(argv[5],"%lf",prho);
    sscanf(argv[6],"%ld",pseed);
  }
  else
  {
    fprintf(stderr, "Wrong command line parameter for algorithm choice: %s!\n", algo);
    exit(EXIT_FAILURE);
  }
}
