#include "solution2.h"

/* Create an empty solution for a problem of size n */
void create_solution (int n, solution_t *px)
{
  px->in_x = bool_alloc(n+1);

  /* Lists of the point in solution x and in the complement N \setminus x */
  px->next = int_alloc(n+1+1);
  px->prev = int_alloc(n+1+1);

  px->D = int_alloc(n+1);

  clean_solution(px,n);
}

/* Deallocate the solution *px */
void destroy_solution (solution_t *px)
{
  px->f = 0;

  px->card_x = 0;

  free(px->in_x);
  px->in_x = NULL;

  free(px->D);
  px->D = NULL;

  px->head_x = 0;
  px->head_notx = 0;
  free(px->next);
  px->next = NULL;
  free(px->prev);
  px->prev = NULL;
}

/* Turn solution *px into the empty set for a problem of size n */
void clean_solution (solution_t *px, int n)
{
  int p;

  px->f = 0;

  px->card_x = 0;

  /* The solution only contains the sentinel */
  px->head_x = 0;
  px->prev[0] = px->next[0] = 0;

  /* The complement list contains all points, sorted by increasing indices */
  px->head_notx = n+1;
  for (p = 1; p <= n; p++)
  {
    px->in_x[p] = false;
    px->D[p] = 0;
    px->prev[p+1] = p;
    px->next[p] = p+1;
  }
  px->prev[1] = n+1;
  px->next[n+1] = 1;
}

/* Copy solution *px_orig into solution *px_dest
   Notice: no check is made on the size of the allocated vectors and on the list heads! */
void copy_solution (solution_t *px_orig, solution_t *px_dest)
{
  int i;

  px_dest->f = px_orig->f;

  px_dest->card_x = px_orig->card_x;

  for (i = px_dest->head_x+1; i <= px_dest->head_notx-1; i++)
    px_dest->D[i] = px_orig->D[i];

  for (i = px_dest->head_x; i <= px_dest->head_notx-1; i++)
    px_dest->in_x[i] = px_orig->in_x[i];

  px_dest->head_x = px_orig->head_x;
  px_dest->head_notx = px_orig->head_notx;

  for (i = px_dest->head_x; i <= px_dest->head_notx; i++)
  {
    px_dest->next[i] = px_orig->next[i];
    px_dest->prev[i] = px_orig->prev[i];
  }
}

/* Print by increasing indices solution *px for a problem of size n */
void print_sorted_solution (solution_t *px, int n)
{
  int i;

  printf("%8d ",get_obj(px));
  for (i = 1; i <= n; i++)
    if (px->in_x[i] == true) printf("%4d ",i);
}

/* Check the internal consistency of solution *px based on instance *pI,
   starting from the incidence vector */
bool check_solution (solution_t *px, data_t *pI)
{
  int i, j;
  int ff, d, cont;
  point p;

  ff = 0;
  for (i = 1; i <= pI->n; i++)
    if (px->in_x[i] == true)
      for (j = i+1; j <= pI->n; j++)
        if (px->in_x[j] == true)
          ff += pI->d[i][j];

  if (ff != px->f) return false;

  for (i = 1; i <= pI->n; i++)
  {
    d = 0;
    for (j = 1; j <= pI->n; j++)
      if (px->in_x[j] == true)
        d += pI->d[i][j];
  
    if (d != px->D[i]) return false;
  }
    
  cont = 0;
  for (p = first_point_in(px); !end_point_list(p,px); p = next_point(p,px))
  {
    if (px->in_x[get_index(p,pI)] == false) return false;
    cont++;
  }
  if (cont != px->card_x) return false;

  cont = 0;
  for (p = first_point_out(px); !end_point_list(p,px); p = next_point(p,px))
  {
    if (px->in_x[get_index(p,pI)] == true) return false;
    cont++;
  }
  if (cont != pI->n-px->card_x) return false;

  return true;
}

/* Get the objective function value for solution *px */
int get_obj (solution_t *px)
{
  return px->f;
}

/* Get the cardinality of solution *px */
int get_card (solution_t *px)
{
  return px->card_x;
}

/* Indicate whether p is a regular point or a sentinel */
bool end_point_list (point p, solution_t *px)
{
  return ( (p <= px->head_x) || (p >= px->head_notx) );
}

/* Return the first and the last point of solution *px */
point first_point_in (solution_t *px)
{
  return px->next[px->head_x];
}

point last_point_in (solution_t *px)
{
  return px->prev[px->head_x];
}

/* Return the first and the last point of the complement of solution *px */
point first_point_out (solution_t *px)
{
  return px->next[px->head_notx];
}

point last_point_out (solution_t *px)
{
  return px->prev[px->head_notx];
}

/* Return the point following or preceding p in solution *px */
point next_point (point p, solution_t *px)
{
  return px->next[p];
}

point prev_point (point p, solution_t *px)
{
  return px->prev[p];
}

/* Add the point i to solution *px */
void add_point (point i, solution_t *px, data_t *pI)
{
  point pi, si, p;
  int id, j;

  id = get_index(i,pI);
  if (px->in_x[id] == true)
  {
    fprintf(stderr,"Point %d is already in the solution!\n",i);
    exit(EXIT_FAILURE);
  }

  /* Remove i from the current list */
  si = px->next[id];
  pi = px->prev[id];
  px->next[pi] = si;
  px->prev[si] = pi;

  /* Add i to list x */
  pi = px->prev[px->head_x];
  si = px->head_x;
  px->next[id] = si;
  px->prev[id] = pi;
  px->next[pi] = id;
  px->prev[si] = id;

  /* Increase the cardinality */
  px->card_x++;

  /* Update the incidence vector */
  px->in_x[id] = true;

  /* Update the objective function value */
  for (p = first_point_in(px); !end_point_list(p,px); p = next_point(p,px))
    px->f += pI->d[get_index(p,pI)][id];

  /* Update the vector of total distances from the solution */
  for (j = 1; j <= pI->n; j++)
    px->D[j] += pI->d[j][id];
}

/* Delete the point of index i from solution *px */
void delete_point (point i, solution_t *px, data_t *pI)
{
  point pi, si, p;
  int id, j;

  id = get_index(i,pI);
  if (px->in_x[id] == false)
  {
    fprintf(stderr,"Point %d is already out of the solution!\n",i);
    exit(EXIT_FAILURE);
  }

  /* Remove i from the current list */
  si = px->next[id];
  pi = px->prev[id];
  px->next[pi] = si;
  px->prev[si] = pi;

  /* Add i to list N \ x */
  pi = px->prev[px->head_notx];
  si = px->head_notx;
  px->next[id] = si;
  px->prev[id] = pi;
  px->next[pi] = id;
  px->prev[si] = id;

  /* Decrease the cardinality */
  px->card_x--;

  /* Update the incidence vector */
  px->in_x[id] = false;

  /* Update the objective function value */
  for (p = first_point_in(px); !end_point_list(p,px); p = next_point(p,px))
    px->f -= pI->d[get_index(p,pI)][id];

  /* Update the vector of total distances from the solution */
  for (j = 1; j <= pI->n; j++)
    px->D[j] -= pI->d[j][id];
}

/* Swap point i (internal) and point j (external) with respect to solution *px */
void swap_points (point i, point j, solution_t* px, data_t* pI)
{
  point pj, sj, pi, si;
  int idj, idi, id;

  idj = get_index(j,pI);
  if (px->in_x[idj] == true)
  {
    fprintf(stderr,"Point %d is already in the solution!\n",j);
    exit(EXIT_FAILURE);
  }

  idi = get_index(i,pI);
  if (px->in_x[idi] == false)
  {
    fprintf(stderr,"Point %d is already out of the solution!\n",i);
    exit(EXIT_FAILURE);
  }

  /* Remove j from the current list */
  sj = px->next[idj];
  pj = px->prev[idj];
  px->next[pj] = sj;
  px->prev[sj] = pj;

  /* Add j to list x */
  pj = px->prev[px->head_x];
  sj = px->head_x;
  px->next[idj] = sj;
  px->prev[idj] = pj;
  px->next[pj] = idj;
  px->prev[sj] = idj;

  /* Remove i from the current list */
  si = px->next[idi];
  pi = px->prev[idi];
  px->next[pi] = si;
  px->prev[si] = pi;

  /* Add i to list N \ x */
  pi = px->prev[px->head_notx];
  si = px->head_notx;
  px->next[idi] = si;
  px->prev[idi] = pi;
  px->next[pi] = idi;
  px->prev[si] = idi;

  /* Update the incidence vector */
  px->in_x[idj] = true;
  px->in_x[idi] = false;

  /* Update the objective function value with the constant-time formula */
  px->f += px->D[idj] - px->D[idi] - pI->d[idj][idi];

  /* Update the vector of total distances from the solution */
  for (id = px->head_x + 1; id <= px->head_notx - 1; id++)
    px->D[id] += pI->d[id][idj] - pI->d[id][idi];
}

/* Compute the total distance of a point i from all point of solution *px */
int dist_from_x (point i, solution_t *px, data_t *pI)
{
  point j;
  int d;

  d = 0;
  for (j = first_point_in(px); !end_point_list(j,px); j = next_point(j,px))
    d += pI->d[get_index(i,pI)][get_index(j,pI)];

  return d;
}
