#ifndef __SOLUTION_H
#define __SOLUTION_H


#include "data.h"


typedef struct solution_s
{
  int f;         /* solution value */

  bool *in_x;    /* incidence vector: in_x[i] = true if i \in x,
                                      in_x[i] = false if i \notin x */

  /* Lists of points in solution x and in the complement N \setminus x */
  int head_x;    /* sentinel of the list of points in x */
  int head_notx; /* sentinel of the list of points in N \setminus x */
  int *next;     /* next     element for each point i in either list */
  int *prev;     /* previous element for each point i in either list */

  int card_x;    /* cardinality of the solution */

} solution_t;


/* Create an empty solution for a problem of size n */
void create_solution (int n, solution_t *px);

/* Deallocate the solution *px */
void destroy_solution (solution_t *px);

/* Turn solution *px into the empty set for a problem of size n */
void clean_solution (solution_t *px, int n);

/* Copy solution *px_orig into solution *px_dest
   Notice: no check is made on the size of the allocated vectors and on the list heads! */
void copy_solution (solution_t *px_orig, solution_t *px_dest);

/* Print by increasing indices solution *px for a problem of size n */
void print_sorted_solution (solution_t* px, int n);

/* Check the internal consistency of solution *px based on instance *pI,
   starting from the incidence vector */
bool check_solution (solution_t *px, data_t *pI);


/* Get the objective function value for solution *px */
int get_obj (solution_t *px);

/* Get the cardinality of solution *px */
int get_card (solution_t *px);

/* Indicate whether p is a regular point or a sentinel */
bool end_point_list (point p, solution_t *px);

/* Return the first and the last point of solution *px */
point first_point_in (solution_t *px);
point last_point_in (solution_t *px);

/* Return the first and the last point of the complement of solution *px */
point first_point_out (solution_t *px);
point last_point_out (solution_t *px);

/* Return the point following and preceding p in solution *px */
point next_point (point p, solution_t *px);
point prev_point (point p, solution_t *px);


/* Functions to manipulate solutions */

/* Add point i to solution *px */
void add_point (point i, solution_t *px, data_t *pI);

/* Delete point i from solution *px */
void delete_point (point i, solution_t *px, data_t *pI);

/* Swap point i (internal) and point j (external) with respect to solution *px */
void swap_points (point i, point j, solution_t* px, data_t* pI);

/* Compute the total distance of a point i from all point of solution *px */
int dist_from_x (point i, solution_t* px, data_t* pI);

#endif /* __SOLUTION_H */
