/// \file solution.h
/// \brief Definition of data structures and functions to handle solutions.
///    

#ifndef _grafo_h
#define _grafo_h

#include "defs.h"

typedef struct _solution_s Solution;
typedef Solution SolutionList;
typedef Solution SolutionPos;

#include "residual_list.h"
#include "hydbond_list.h"
#include "cluster_list.h"

/// \brief Data structure representing a solution
///
/// A solution is characterized by a unique numerical index, the number of atoms
/// and hydrogen bonds it includes, the number and the list of residuals, the
/// number and the list of residual clusters. It also has pointers to link it
/// into a solution list.

struct _solution_s
{
  int Id;           ///<  Solution identifier 

  int NumAtoms;     ///<  Number of atoms included in the solution
  int NumHydBonds;  ///<  Number of hydrogen bonds included in the solution

  int NumResiduals; ///<  Number of residuals included in the solution
  ResidualList *RL; ///<  List of residuals included in the solution

  int NumClusters;  ///<  Number of atoms included in the solution
  ClusterList  *CL; ///<  Number of atoms included in the solution

  Solution *next;   ///< Pointer to the next     element in the list to which the solution belongs (NULL if none)
  Solution *prev;   ///< Pointer to the previous element in the list to which the solution belongs (NULL if none)
};


/// Create an empty solution
Solution *createSolution ();

/// Destroy a given solution
void destroySolution (Solution **pS);

/// Load from InputFile the atoms and group them into residuals.
/// Return the number of atoms loaded.
int LoadAtoms (char *InputFile, Solution *S);

/// Load from InputFile the atoms and copy them into a vector.
/// Return the number of atoms loaded.
/// Note: it requires the vector from outside; 
/// therefore, one needs to know in advance the number of atoms.
int LoadAtomVector (Atom *AtomVector, char *InputFile);

/// Insert residual R into the list of residuals of solution S.
/// It increases NumResiduals, without checking whether residual R (or an identical one)
/// already belongs to S.
void InsResidualSolution (Residual *R, Solution *S);

/// Insert atom A into the correct list of atoms of residual R,
/// depending on the chemical element of A (acceptors/donors, hydrogens, others).
void InsAtomResidual (Atom *A, Residual *R);

/// Determine all the hydrogen bonds of solution S between residuals in first shell and residuals
/// in first or second shell. The first and second shell are, respectively, defined as the space 
/// regions at a distance lower than DELTA_SHELL (respectively, DELTA_SHELL2) from any atom of a 
/// solute residual. The solute is any substance different from water.
/// A hydrogen bond exists if:
/// 1) the angle between the hydrogen-donor and the hydrogen acceptor directions has a cosine lower ALFA
/// 2) the distance between the donor and the acceptor is lower than DELTA
void DetermineHydBonds (Solution *S, double ALFA, double DELTA, double DELTA_SHELL, double DELTA_SHELL2);

/// Scan a given list BL of hydrogen bonds and delete all the ones which do not exist,
/// given the spatial positions of the atoms contained in AtomVector.
/// A hydrogen bond exists if
/// 1) the angle between the hydrogen-donor and the hydrogen acceptor directions has a cosine lower ALFA
/// 2) the distance between the donor and the acceptor is lower than DELTA
int FilterRemainingHydBonds (HydBondList *BL, Atom *AtomVector, double ALPHA, double DELTA);

/// Create a copy of the list of hydrogen bonds of solution S.
HydBondList *CopyHydBondList (Solution *S);

/// Determine the clusters of residuals linked by hydrogen bonds in solution S 
/// and add to S the number of residuals (NumResiduals) and their list. 
void DetermineClusters (Solution *S);

/// Write a verbose description of solution S into the text file LogFile.
void WriteLog (Solution *S, char *LogFile);

/// Save solution S in file OutputFile, in PDB format.
void WritePdbSolution (Solution *S, char *OutputFile);

//void FrequenzaCluster(Grafo *G, int SOGLIA, char *FrequencyFile);

/// Returns TRUE if residual R of solution S belongs to the first shell, FALSE otherwise.
/// A residual belongs to the first shell if it has a distance lower than DELTA_SHELL
/// from any solute residual. Solute residuals obviously belong to the first shell.
boolean BelongsToShell (Solution *S, Residual *R, double DELTA_SHELL);

/// Return a list containing copies of all solute residuals in solution S.
int FindSoluteResiduals (Solution *S, ResidualList *RL);

/// Return a list containing copies of all water residuals in solution S belonging to the first shell.
/// A residual belongs to the first shell if it has a distance lower than DELTA_SHELL
/// from any solute residual. 
int FindFirstShellWaterResidual (Solution *S, ResidualList *WL, double DELTA_SHELL);

#endif
