/// \file solution.c
/// \brief Declaration of data structures and functions to handle solutions.
///    

#include "solution.h"

Solution *createSolution ()
{
  Solution *S;

  S = (Solution *) malloc(sizeof(Solution));
  if (S == NULL)
  {
    printf("Not enough memory to allocate a solution!\n");
    exit(EXIT_MEMORY);
  }

  S->NumAtoms     = 0;
  S->NumResiduals = 0;
  S->NumHydBonds  = 0;
  S->NumClusters  = 0;
  S->RL = (ResidualList *) createResidualList();
  S->CL = (ClusterList *)  createClusterList();

  return S;
}


void destroySolution (Solution **pS)
{
  Solution *S;
  ResidualPos *pR;
  ClusterPos *pC;

  S = *pS;

  // Destroy the residual list
  for (pR = firstResidualList(S->RL); !endResidualList(pR,S->RL); pR = nextResidualList(pR))
    emptyResidual(pR);
  destroyResidualList(&S->RL);

  // Destroy the cluster list and the lists of the pointers to the residuals for each cluster
  for (pC = firstClusterList(S->CL); !endClusterList(pC,S->CL); pC = nextClusterList(pC))
    destroypResidualList(&pC->PRL);
  destroyClusterList(&S->CL);

  S->NumAtoms     = 0;
  S->NumResiduals = 0;
  S->NumHydBonds  = 0;
  S->NumClusters  = 0;

  free(S);
  *pS = NULL;
}


int LoadAtoms (char *InputFile, Solution *S)
{
  FILE *fInputFile;

  // Variabili in cui si memorizzano i dati letti per ogni atomo
  int atom_id, residual_id;
  char atom_element[4];
  char residual_substance[4];
  double x, y, z;
  
  Residual *R; // Current residual in the solution
  Atom *A;     // Current atom
  
  
  fInputFile = fopen(InputFile,"r");
  if (fInputFile == NULL)
  {
    printf("File %s could not be opened!\n",InputFile);
    exit(EXIT_OPENFILE);
  }

  R = NULL;
  while(!feof(fInputFile))
  {
    if (fscanf(fInputFile,"ATOM %d %s %s %d %lf %lf %lf 0.00 0.00\n",&atom_id,atom_element,
                          residual_substance,&residual_id,&x,&y,&z) != 7)
    {
      printf("Wrong format in input file for atom %d!\n",S->NumAtoms+1);
      exit(EXIT_WRONGINPUTFORMAT);
    }
    else
    { 
      S->NumAtoms++;

      // Create a new atom
      A = createAtom(atom_id,x,y,z,atom_element);

      // Si presuppone che gli atomi del file di input siano ordinati 
      // per valori crescenti dell'id delle molecole
      if ((R == NULL) || (residual_id != R->Id)) //Se la molecola dell'atomo non esiste ancora viene creata
      {
        R = createResidual(residual_id,residual_substance);
        //Inserisce la molecola creata nel grafo - vedi grafo.h
        InsResidualSolution(R,S);
      }

       //Inserisce l'atomo che  appena stato creato nella molecola
      InsAtomResidual(A,R);
    }
  }
  fclose(fInputFile);

  return S->NumAtoms;
}


int LoadAtomVector (Atom *AtomVector, char *InputFile)
{
  FILE *fInputFile;

  // Variabili in cui si memorizzano i dati letti per ogni atomo
  int atom_id, residual_id;
  char atom_element[4];
  char residual_substance[4];
  double x, y, z;
  int atom_count;
  
  
  fInputFile = fopen(InputFile,"r");
  if (fInputFile == NULL)
  {
    printf("File %s could not be opened!\n",InputFile);
    exit(EXIT_OPENFILE);
  }

  atom_count = 0;
  while(!feof(fInputFile))
  {
    if (fscanf(fInputFile,"ATOM %d %s %s %d %lf %lf %lf 0.00 0.00\n",&atom_id,atom_element,
                          residual_substance,&residual_id,&x,&y,&z) != 7)
    {
      printf("Wrong format in input file for atom %d!\n",atom_count+1);
      exit(EXIT_WRONGINPUTFORMAT);
    }
    else
    { 
      atom_count++;

      AtomVector[atom_id].Id = atom_id;
      strcpy(AtomVector[atom_id].Element,atom_element);
      AtomVector[atom_id].x = x;
      AtomVector[atom_id].y = y;
      AtomVector[atom_id].z = z;
      AtomVector[atom_id].next = AtomVector[atom_id].prev = NULL;
      AtomVector[atom_id].R = NULL;
     }
  }
  fclose(fInputFile);

  return atom_count;
}


void InsResidualSolution (Residual *R, Solution *S)
{
    //Inserisce la molecola in fondo alla lista delle molecole del grafo G
    appendResidualList(R,S->RL);

    // Incrementa il contatore
    S->NumResiduals++;
}


/**
  Insert an atom into a residual
@param[in]     A The new atom to be inserted
@param[in,out] R The residual in which A is inserted
@remarks It does not check whether the atom already belongs to R
*/

void InsAtomResidual (Atom *A, Residual *R)
{
    /*ATTENZIONE - prima di effettuare l'inserimento NON viene controllato
    se l'atomo esiste gi prima dell'inserimento*/

    /*Se si tratta di idrogeno l'atomo viene aggiunto in coda
    alla lista degli idrogeni*/
  // ATTENZIONE: SAREBBE MEGLIO USARE STRCMP, MA PRIMA BISOGNA RISOLVERE IL PROBLEMA
  // DEI SUFFISSI (H1 E H2 AL POSTO DI H), CHE COMPORTA DI CARICARE 
  // I DATI IN MODO PIU' COMPLESSO
    if (A->Element[0] == 'H')
    {
      appendAtomList(A,R->HL);
      A->R = R;
    }
    /*Se si tratta di AZOTO (N) o OSSIGENO (O) l'atomo viene aggiunto in coda
    alla lista dei donori e accettori */
    else if (A->Element[0]=='N' || A->Element[0]=='O')
    {
      appendAtomList(A,R->ADL);
      A->R = R;
    }
    /*in tutti gli altri casi l'atomo viene aggiunto in coda alla lista degli
    altri atomi*/
    else
    {
      appendAtomList(A,R->OL);
      A->R = R;
    }
}


// Compute the distance between atoms A1 and A2
double Distance (Atom *A1, Atom *A2)
{
  double Ris;
  Ris = sqrt( (A1->x - A2->x) * (A1->x - A2->x) + (A1->y - A2->y) * (A1->y - A2->y) + (A1->z - A2->z) * (A1->z - A2->z) );
  return Ris;
}


// Compute the cosine of the angle formed by D H A
double Angle (Atom *D, Atom *H, Atom *A) 
{
  double Ris, HD, HA, ProdScal;

  HD = Distance(D,H);
  HA = Distance(A,H);
  ProdScal = ((D->x - H->x)*(A->x - H->x) + (D->y - H->y)*(A->y - H->y) + (D->z - H->z)*(A->z - H->z));
  Ris = ProdScal/(HD*HA);
  return Ris;
}


// Determines whether residual R belongs to the "first shell" surrounding the solute

boolean BelongsToShell (Solution *S, Residual *R, double DELTA_SHELL)
{
  Residual *R1;
  Atom *H, *D, *A, *H1, *D1, *A1;


  if (strcmp(R->Substance, "WAT") != 0) return TRUE;

  for (R1 = firstResidualList(S->RL); !endResidualList(R1, S->RL); R1 = nextResidualList(R1))
  {
    if ((strcmp(R1->Substance, "WAT") != 0) && (R1 != R))
    {
      // per gli atomi di idrogeno delle molecole di soluto
      for (H1 = firstAtomList(R1->HL); !endAtomList(H1,R1->HL); H1 = nextAtomList(H1))
      {
        for(H = firstAtomList(R->HL); !endAtomList(H,R->HL); H = nextAtomList(H))
          if ((Distance(H1,H)) <= DELTA_SHELL) return TRUE;

        for (D = firstAtomList(R->ADL); !endAtomList(D,R->ADL); D = nextAtomList(D))
          if ((Distance(H1, D))<=DELTA_SHELL) return TRUE;
     
        for (A = firstAtomList(R->OL); !endAtomList(A,R->OL); A = nextAtomList(A))
          if ((Distance(H1, A))<=DELTA_SHELL) return TRUE;
      }

      // per gli atomi donatori ed accetori del soluto
      for(D1 = firstAtomList(R1->ADL); !endAtomList(D1,R1->ADL); D1 = nextAtomList(D1))
      {
        for(H = firstAtomList(R->HL); !endAtomList(H,R->HL); H = nextAtomList(H))
          if ((Distance(D1,H)) <= DELTA_SHELL) return TRUE;
     
        for (D = firstAtomList(R->ADL); !endAtomList(D,R->ADL); D = nextAtomList(D))
          if ((Distance(D1,D)) <= DELTA_SHELL) return TRUE;
     
        for (A = firstAtomList(R->OL); !endAtomList(A,R->OL); A = nextAtomList(A))
          if ((Distance(D1,A)) <= DELTA_SHELL) return TRUE;
      }
   
      // per gli atomi altri del soluto
      for (A1 = firstAtomList(R1->OL); !endAtomList(A1,R1->OL); A1 = nextAtomList(A1))
      {
        for (H = firstAtomList(R->HL); !endAtomList(H,R->HL); H = nextAtomList(H))
          if ((Distance(A1,H)) <= DELTA_SHELL) return TRUE;
     
        for (D = firstAtomList(R->ADL); !endAtomList(D,R->ADL); D = nextAtomList(D))
          if ((Distance(A1,D)) <= DELTA_SHELL) return TRUE;
     
        for (A = firstAtomList(R->OL); !endAtomList(A,R->OL); A = nextAtomList(A))
          if ((Distance(A1,A)) <= DELTA_SHELL) return TRUE;
      }
    }
  }
  return FALSE;
}


void insBondSolution (Solution *S, Residual *R, Residual *R1, Atom *D, Atom *H, Atom *A, double Dist, double Ang)
{
  HydBond *L1, *L2;

  L1 = createHydBond(S->NumHydBonds+1,Dist,Ang,A,D,H,R,R1);
  appendHydBondList(L1,R->BL);
  R->NumHydBonds++;

    // ATTENZIONE. QUI NON VA BENE; IL LEGAME VIENE ATTACCATO SOLO A UNA DELLE DUE MOLECOLE
    // QUINDI E' ORIENTATO. MA COSI' L'ALGORITMO PER LE COMPONENTI CONNESSE NON FUNZIONA!!!
    // INFATTI RISULTANO MOLTE PIU' COMPONENTI DI PRIMA!

  L2 = createHydBond(S->NumHydBonds+1,Dist,Ang,A,D,H,R,R1);
  appendHydBondList(L2,R1->BL);
  R1->NumHydBonds++;

  L1->opposite = L2;
  L2->opposite = L1;

  S->NumHydBonds++;

}


void DetermineHydBonds (Solution *S, double ALFA, double DELTA, double DELTA_SHELL, double DELTA_SHELL2)
{
  Residual *R, *R1;
  Atom *H, *D, *A;
  double Dist, Ang;
  int i=0,j=0,k=0;


  for (R = firstResidualList(S->RL); !endResidualList(R,S->RL); R = nextResidualList(R))
  { 
    i++;
    // Ignore all residuals out of the second shell
    if (BelongsToShell(S,R,DELTA_SHELL2))
    {
      j++;
      
      for (H = firstAtomList(R->HL); !endAtomList(H,R->HL); H = nextAtomList(H))
        for (D = firstAtomList(R->ADL); !endAtomList(D,R->ADL); D = nextAtomList(D))
          for (R1 = firstResidualList(S->RL); !endResidualList(R1,S->RL); R1 = nextResidualList(R1))
            if ((BelongsToShell(S,R1,DELTA_SHELL)) ||
                ((BelongsToShell(S,R1,DELTA_SHELL2)) && (BelongsToShell(S,R, DELTA_SHELL))))
            {
              if (R != R1)
                for(A = firstAtomList(R1->ADL); !endAtomList(A,R1->ADL); A=nextAtomList(A))
                {
                  Dist = Distance(D,A);//calcola Distance donore accettore
                  Ang = Angle(D,H,A);//Calcola l'angolo tra D,H,A
                  if ((Ang <= ALFA) && (Dist <= DELTA))
                  {
                    insBondSolution(S,R,R1,D,H,A,Dist,Ang);
                    printf(".");
                  }
                }
            }
    }
  }
  printf("\n\nTotal Residue Number:%d \n2nd Shell Residue Number:%d \n", i, j);
  printf("%d bonds in 1st shell or between 1st and 2nd shell\n", S->NumHydBonds);
}


int FilterRemainingHydBonds (HydBondList *BL, Atom *AtomVector, double ALPHA, double DELTA)
{
  HydBondPos *pB;
  unsigned int i, Id_don, Id_acc, Id_H;
  double dist, ang;


  i = 0;
  for (pB = firstHydBondList(BL); !endHydBondList(pB,BL); )
  {
    Id_don = pB->D->Id;
    Id_acc = pB->A->Id;
    Id_H   = pB->H->Id;

    dist = Distance(&AtomVector[Id_acc],&AtomVector[Id_don]);
    ang  = Angle(&AtomVector[Id_don],&AtomVector[Id_H],&AtomVector[Id_acc]);

    if ( (dist <= DELTA) && (ang <= ALPHA))
    {
      i++;
      pB = nextHydBondList(pB);
    }
    else
    {
      cancHydBondList(&pB);
    }
  }

 return i;
}


// Create a list with copies of the current hydrogen bonds
HydBondList *CopyHydBondList (Solution *S)
{
  HydBondList *BL;
  ResidualPos *pR;
  HydBondPos *pB;
  HydBond *B;

  BL = createHydBondList();
  for (pR = firstResidualList(S->RL); !endResidualList(pR,S->RL); pR = nextResidualList(pR))
    for (pB = firstHydBondList(pR->BL); !endHydBondList(pB,pR->BL); pB = nextHydBondList(pB))
      // Each hydrogen bond appears both in the list of the donor residual and the acceptor residual.
      // To avoid counting it twice, we only copy the occurrence in the donor residual.
      if (pB->DonRes->Id == pR->Id)
      {
        B = createHydBond(pB->Id,pB->Distance,pB->Angle,pB->A,pB->D,pB->H,pB->DonRes,pB->AccRes);
        appendHydBondList(B,BL);
      }

  return BL;
}


void VisitCluster (Solution *S, Residual *R, Cluster *C)
{
  HydBondPos *pB;
  pResidual *pR;


  pR = createpResidual(R);
  appendpResidualList(pR,C->PRL);
  C->NumResiduals++;

  // Incrementare il contatore delle molecole di acqua o di soluto, secondo il tipo di R
  if (strcmp(pR->R->Substance,"WAT") != 0)
    C->NumSolvents++;
  else
    C->NumSolutes++;

  R->C = C;
  for (pB = firstHydBondList(R->BL); !endHydBondList(pB,R->BL); pB = nextHydBondList(pB))
  {
    // One of the residuals involved in the hydrogen bond is R, already visited.
    // The other one could be new: if it is, proceed with the visit.
    if ( (pB->DonRes != R) && (pB->DonRes->C == NULL) ) VisitCluster(S,pB->DonRes,C);
    if ( (pB->AccRes != R) && (pB->AccRes->C == NULL) ) VisitCluster(S,pB->AccRes,C);
  }
}


void DetermineClusters (Solution *S)
{
  Cluster *C;
  Residual *R;

  for (R = firstResidualList(S->RL); !endResidualList(R,S->RL); R = nextResidualList(R))
  {
    if (R->C == NULL)
    {
      C = createCluster(S->NumClusters+1);
      appendClusterList(C,S->CL);
      S->NumClusters++;
      VisitCluster(S,R,C);
    }
  }
}


void WriteLog (Solution *S, char *LogFile)
{
  ResidualPos *pR;
  AtomPos *pA;
  HydBondPos *pB;
  ClusterPos *pC;
  pResidualPos *ppR;
  FILE *fLogFile;

  
  fLogFile = fopen(LogFile,"w");
  if (fLogFile == NULL)
  {
    printf("File %s could not be opened!\n",LogFile);
    exit(EXIT_OPENFILE);
  }

  fprintf(fLogFile,"The solution includes %d residuals\n",S->NumResiduals);
  fprintf(fLogFile,"The solution includes %d hydrogen bonds\n",S->NumHydBonds);
  fprintf(fLogFile,"The solution includes %d cluster\n",S->NumClusters);
  
  fprintf(fLogFile,"---- CLUSTERS ----\n");
  for (pC = firstClusterList(S->CL); !endClusterList(pC,S->CL); pC = nextClusterList(pC))
  {
    fprintf(fLogFile,"Cluster %d contains %d residuals\n",pC->Id,pC->NumResiduals);
    fprintf(fLogFile, "(%d solvent and %d solute residuals)\n",pC->NumSolvents, pC->NumSolutes);
    for (ppR = firstpResidualList(pC->PRL); !endpResidualList(ppR,pC->PRL); ppR = nextpResidualList(ppR))
    {
      pR = ppR->R;

      fprintf(fLogFile,"\tResidual %d %s\n",pR->Id,pR->Substance);

      // Print the atoms included in the current residual
      for (pA = firstAtomList(pR->HL); !endAtomList(pA,pR->HL); pA = nextAtomList(pA))
        fprintf(fLogFile,"\tAtom %d: %.2s is in (%.3f,%.3f,%.3f)\n", pA->Id, pA->Element, pA->x, pA->y, pA->z);

      for (pA = firstAtomList(pR->ADL); !endAtomList(pA,pR->ADL); pA = nextAtomList(pA))
        fprintf(fLogFile,"\tAtom %d: %.2s is in (%.3f,%.3f,%.3f)\n", pA->Id, pA->Element, pA->x, pA->y, pA->z);

      for (pA = firstAtomList(pR->OL); !endAtomList(pA,pR->OL); pA = nextAtomList(pA))
        fprintf(fLogFile,"\tAtom %d: %.2s is in (%.3f,%.3f,%.3f)\n", pA->Id, pA->Element, pA->x, pA->y, pA->z);

      // Print the hydrogen bonds in which the current residual is involved
      for (pB = firstHydBondList(pR->BL); !endHydBondList(pB,pR->BL); pB = nextHydBondList(pB))
        fprintf(fLogFile,"\t\tHydrogen bond %d:  dist: %lf  angle: %lf  donor: %d (%.2s)  hydrogen: %d (%.2s) acceptor: %d (%.2s)\n",
                pB->Id,pB->Distance,pB->Angle,pB->D->Id,pB->D->Element,pB->H->Id,pB->H->Element,pB->A->Id,pB->A->Element);
    }
    fprintf(fLogFile,"\n");
  }
 
  fclose(fLogFile);
}


void WritePdbSolution (Solution *S, char *OutputFile)
{
  FILE *fOutputFile; 
  ClusterPos *pC;
  pResidualPos *ppR;
  ResidualPos *pR;
  AtomPos *pA;
  HydBondPos *pB;
  int n;
  
  
  fOutputFile = fopen(OutputFile,"w");
  if (fOutputFile == NULL)
  {
    printf("File %s could not be opened!\n",OutputFile);
    exit(EXIT_OPENFILE);
  }

  // Write in the output file, in format PDB, that is as REMARKs, the residuals belonging to each cluster
  n = 1;
  for (pC = firstClusterList(S->CL); !endClusterList(pC,S->CL); pC = nextClusterList(pC))
  {
    fprintf(fOutputFile,"REMARK 4   CLUSTER %d MOL_INIZ %d",pC->Id,n);
    for (ppR = firstpResidualList(pC->PRL); !endpResidualList(ppR,pC->PRL); ppR = nextpResidualList(ppR))
      n++;
    fprintf(fOutputFile," MOL_FINALE %d\n",n-1);
  }

  // Write the atoms, grouped by clusters and residuals, in the output file in PDB format 
  n = 1;
  for (pC = firstClusterList(S->CL); !endClusterList(pC,S->CL); pC = nextClusterList(pC))
  {
    for (ppR = firstpResidualList(pC->PRL); !endpResidualList(ppR,pC->PRL); ppR = nextpResidualList(ppR))
    {
      pR = ppR->R;
      for (pA = firstAtomList(pR->ADL); !endAtomList(pA,pR->ADL); pA = nextAtomList(pA))
        fprintf(fOutputFile,"ATOM  %5d  %-3s %-4s %4d    %8.3f%8.3f%8.3f %d\n",pA->Id,pA->Element,pR->Substance,n,pA->x,pA->y,pA->z,pR->Id);
      for (pA = firstAtomList(pR->HL); !endAtomList(pA,pR->HL); pA=nextAtomList(pA))
        fprintf(fOutputFile,"ATOM  %5d  %-3s %-4s %4d    %8.3f%8.3f%8.3f %d\n",pA->Id,pA->Element,pR->Substance,n,pA->x,pA->y,pA->z,pR->Id);
      for (pA = firstAtomList(pR->OL); !endAtomList(pA,pR->OL); pA=nextAtomList(pA))
        fprintf(fOutputFile,"ATOM  %5d  %-3s %-4s %4d    %8.3f%8.3f%8.3f %d\n",pA->Id,pA->Element,pR->Substance,n,pA->x,pA->y,pA->z,pR->Id);
      n++;
    }
  }

  // Write the list of the hydrogen bonds in the output file in PDB format
  for (pR = firstResidualList(S->RL); !endResidualList(pR,S->RL); pR = nextResidualList(pR))
    for (pB = firstHydBondList(pR->BL); !endHydBondList(pB,pR->BL); pB = nextHydBondList(pB))
      fprintf(fOutputFile,"HYDBND       %-3s %-3s  %5d  %-3s    %5d   %-3s %-3s  %5d\n",
              pB->A->Element,pB->DonRes->Substance,pB->DonRes->Id,pB->H->Element,pB->DonRes->Id,pB->D->Element,
              pB->AccRes->Substance,pB->AccRes->Id);

  fclose(fOutputFile);
}


// Preleva dalla lista delle molecole solamente le molecole di soluto
int FindSoluteResiduals (Solution *S, ResidualList *RL)
{
  char sKey[] = "WAT";  
  Residual *R1, *R2;
  int cont;

  cont = 0;
  for(R1 = firstResidualList(S->RL); !endResidualList(R1, S->RL); R1 = nextResidualList(R1))
  {
    if(strcmp(R1->Substance, sKey) != 0)
    {
      R2 = createResidual(R1->Id, R1->Substance);
      R2->NumHydBonds = R1->NumHydBonds;
      R2->BL = R1->BL;
      R2->ADL = R1->ADL;
      R2->HL = R1->HL;
      R2->OL = R1->OL;
      R2->next = R1->next;
      R2->prev = R1->prev;
      R2->C = R1->C;
      appendResidualList(R2, RL);
      cont++;
    }
  }

  return cont;
}


// Find water residual in 1 shell
int FindFirstShellWaterResidual (Solution *S, ResidualList *WL, double DELTA_SHELL)
{
  int NumberResidualFirstShell = 0;
  Residual *R1, *R2;


  // Preleva tutte le molecole d'acqua presenti nella prima shell
  for(R2 = firstResidualList(S->RL); !endResidualList(R2, S->RL); R2 = nextResidualList(R2))
  {
    if(strcmp(R2->Substance, "WAT") == 0)
    {
      if (BelongsToShell(S, R2, DELTA_SHELL) == TRUE)
      {
        NumberResidualFirstShell++;

        R1 = createResidual(R2->Id, R2->Substance);
        R1->NumHydBonds = R2->NumHydBonds;
        R1->BL = R2->BL;
        R1->ADL = R2->ADL;
        R1->HL = R2->HL;
        R1->OL = R2->OL;
        R1->next = R2->next;
        R1->prev = R2->prev;
        R1->C = R2->C;
        appendResidualList(R1, WL);
      }
    }
  }

  return NumberResidualFirstShell;  
}


