/// \file hydbond.h
/// \brief Definition of data structures and functions to handle hydrogen bonds.
///    

#ifndef _hydbond_h
#define _hydbond_h

typedef struct _hydbond_s HydBond;
typedef HydBond HydBondList;
typedef HydBond HydBondPos;

#include "atom.h"
#include "residual.h"

/// \brief Data structure representing a hydrogen bond
///
/// Note: a single hydrogen bond corresponds to two separate data structures: each one 
///       belongs to the hydrogen bond list of one of the two residuals involved in the bond. 
///       The two structures point each other through the field "opposite".

struct _hydbond_s
{
    int Id;          ///< Numerical index of the hydrogen bond
    double Distance; ///< Distance (in Angstrom) between the donor and the acceptor atom involved in the bond
    double Angle;    ///< Cosine of the angle between the hydrogen-donor and the hydrogen-acceptor directions involved in the bond

    Atom *A;   ///< Pointer to the acceptor atom involved in the bond
    Atom *D;   ///< Pointer to the donor    atom involved in the bond
    Atom *H;   ///< Pointer to the hydrogen atom involved in the bond
    Residual *DonRes;  ///< Pointer to the donor    residual involved in the bond
    Residual *AccRes;  ///< Pointer to the acceptor residual involved in the bond

    HydBond *opposite; ///< Pointer to the structure which represents the same hydrogen bond in the bond list of the other residual
    HydBond *next;  ///< Pointer to the next     element in the list to which the hydrogen bond belongs (NULL if none)
    HydBond *prev;  ///< Pointer to the previous element in the list to which the hydrogen bond belongs (NULL if none)
};

/// Create a new hydrogen bond structure, with a given index, distance, angle, linking given atoms and residuals

HydBond *createHydBond (int Id, double Distance, double Angle, Atom *A, Atom *D, Atom *H, Residual *DonRes, Residual *AccRes);

#endif
