/// \file hb-time.c
/// \brief Main file for the computation of the decay profile of hydrogen bonds
/// in a given sequence of shapshots

#include "defs.h"
#include "solution.h"


// Parse the command line
void ReadCommandLine (int argc, char *argv[], char *Prefix, int *suffisso_inizio, int *suffisso_fine, char *OutputFile);

//Legge il file di configurazione
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA);

int *TrackHydBondDecay (Solution *S, char *PrefixFile, unsigned int ti, unsigned int tf,
                        double ALPHA, double DELTA);

void SaveHydBondProfile (int *HydBondProfile, int ti, int tf, char *FreqFile);


int main(int argc, char *argv[])
{
  char Prefix[LUNGHEZZA];     // Name of the PDB input file (without the extension)
  unsigned int ti;            // Starting time of the sequence of snapshots
  unsigned int tf;            // Ending time of the sequence of snapshots
  char InputFile[LUNGHEZZA]; // Name of the PDB input file
  char OutputFile[LUNGHEZZA]; // Name of the output file reporting the hydrogen bond decay over time

  double ALPHA, DELTA, DELTA_SHELL, DELTA_SHELL2;
  unsigned int t;
  int SOGLIA;

  Solution *S;
  int *HydBondProfile;

  FILE *Results;


  // Read the command line
  ReadCommandLine(argc,argv,Prefix,&ti,&tf,OutputFile);

  // Read the configuration file
  ReadConfigurationFile(&ALPHA, &DELTA, &DELTA_SHELL, &DELTA_SHELL2, &SOGLIA);

  // Build the name of the starting input file (nomefileinput.1)
  sprintf(InputFile,"%s.%u",Prefix,ti);

  // Create an empty solution
  S = createSolution();

  // Load the atoms from InputFile into solution S
  printf("Loading the atom positions from %s...",InputFile);
  LoadAtoms(InputFile,S);
  printf("%d atoms\n",S->NumAtoms);

  // Determine the hydrogen bonds in the solution
  printf("Creating the hydrogen bonds for solution %s...\n",InputFile);
  DetermineHydBonds(S,ALPHA,DELTA,DELTA_SHELL,DELTA_SHELL2);

  // Tracking the hydrogen bond decay
  printf("Tracking the hydrogen bond decay from snapshot %d to snapshot %d...\n\n", ti, tf);
  HydBondProfile = TrackHydBondDecay(S, Prefix, ti, tf, ALPHA, DELTA);

  // Inizializzo il file contenente i risultati delle elaborazioni
  Results = fopen(OutputFile, "w");
  if (Results == NULL)
  {
    printf("File %s could not be opened!\n",OutputFile);
    exit(EXIT_OPENFILE);
  }
  fprintf(Results, "Tracking the hydrogen bond decay from snapshot %d to snapshot %d...\n\n", ti, tf);
  for(t = ti; t <= tf; t++)
    fprintf(Results, "Hydrogen bond at snapshot %d: %d\n", t, HydBondProfile[t-ti]);
  fclose(Results);

  // Destroy the data structures representing the solution
  destroySolution(&S);

  return EXIT_SUCCESS;
}


// Read the command line to get the parameters:
// PrefixFile is...

void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile)
{
  if (argc != 5)
  {
    printf("The command line has a wrong format!\n");
    printf("Use: %s [input_file_prefix] [start_suffix] [end_suffix] [output_file]\n",argv[0]);
    exit(EXIT_WRONGCOMMANDLINE);
  }

  strcpy(Prefix,argv[1]);
  *pti = atoi(argv[2]);
  *ptf = atoi(argv[3]);
  strcpy(OutputFile,argv[4]);

  printf("\n- PARAMETERS  -\n");
  printf("   snapshots prefix : %s \n", Prefix);
  printf("   first snapshot   : %d \n", *pti);
  printf("   last snapshot    : %d \n", *ptf);
  printf("   output file name : %s \n\n", OutputFile);
}


// Read from the configuration file "configurazione.txt" the values of ALPHA, DELTA, ...
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA)
{
  FILE *fConfigFile;
  char Parametro[100];
  double Valore;


  fConfigFile = fopen("configurazione.txt","r");
  if (fConfigFile == NULL)
  {
    printf("Configurazione File Incorrect!\n");
    exit(EXIT_OPENFILE);
  }

  while(!feof(fConfigFile))
  {
    fscanf(fConfigFile,"%s %lf\n", Parametro, &Valore);

    if (Parametro[0] == '#')
      continue;
    else
    {
      if (!strcmp(Parametro,"ALPHA")) *ALPHA = Valore;
      if (!strcmp(Parametro,"DELTA")) *DELTA = Valore;
      if (!strcmp(Parametro,"DSHELL")) *DELTA_SHELL = Valore;
      if (!strcmp(Parametro,"DSHELL2")) *DELTA_SHELL2 = Valore;
      if (!strcmp(Parametro,"SOGLIA")) *SOGLIA = (int) Valore;

    }
  }

  fclose(fConfigFile);
}


int *TrackHydBondDecay (Solution *S, char *PrefixFile, unsigned int ti, unsigned int tf,
                        double ALPHA, double DELTA)
{
  int *HydBondProfile;
  Atom *AtomVector;
  unsigned int t;
  char InputFile[LUNGHEZZA];
  HydBondList *BL;


  HydBondProfile = (int *) calloc(tf-ti+1,sizeof(int));
  if (HydBondProfile == NULL)
  {
    printf("Not enough memory to allocate a vector to keep the hydrogen bond decay profile!\n");
    exit(EXIT_MEMORY);
  }

  // Create a vector of atoms to host the data of the subsequent snapshots
  AtomVector = (Atom *) calloc(S->NumAtoms+1,sizeof(Atom));
  if (AtomVector == NULL)
  {
    printf("Not enough memory to allocate a vector of atoms!\n");
    exit(EXIT_MEMORY);
  }

  // Create a list with copies of the current hydrogen bonds
  BL = CopyHydBondList(S);

  HydBondProfile[0] = S->NumHydBonds;
  for (t = ti+1; t <= tf; t++)
  {
    sprintf(InputFile,"%s.%u",PrefixFile,t);
   LoadAtomVector(AtomVector,InputFile);

    // interfaccia il vettore con la lista unica di legami
   HydBondProfile[t-ti] = FilterRemainingHydBonds(BL,AtomVector,ALPHA,DELTA);
  }
  free(AtomVector);
  destroyHydBondList(&BL);

  return HydBondProfile;
}

