/// \file hb-cluster.c
/// \brief Main file for the computation of the clusters
///

#include "defs.h"
#include "solution.h"


void ReadCommandLine (int argc, char *argv[], char *InputFile, char *pOutputType, char *OutputFile);

//Legge il file di configurazione
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA);


int main(int argc, char *argv[])
{
  char InputFile[LUNGHEZZA];  // Name of the PDB input file
  char OutputFile[LUNGHEZZA]; // Name of the output file
  char OutputType;            // Type of the output: PDB file (-pdb) or log file (-log)

  double ALPHA, DELTA, DELTA_SHELL, DELTA_SHELL2;
  int SOGLIA;

  Solution *S;


  // Read the command line
  ReadCommandLine(argc,argv,InputFile,&OutputType,OutputFile);

  // Read the configuration file
  ReadConfigurationFile(&ALPHA, &DELTA, &DELTA_SHELL, &DELTA_SHELL2, &SOGLIA);

  // Create an empty solution
  S = createSolution();

  // Load the atoms from InputFile into solution S
  printf("Loading the atom positions from %s...",InputFile);
  LoadAtoms(InputFile,S);
  printf("%d atoms\n",S->NumAtoms);

  // Determine the hydrogen bonds in the solution
  printf("Creating the hydrogen bonds for solution %s...\n",InputFile);
  DetermineHydBonds(S,ALPHA,DELTA,DELTA_SHELL,DELTA_SHELL2);

  // Determine the clusters of residuals tied by hydrogen bonds
  printf("Creating the clusters for solution %s...",InputFile);
  DetermineClusters(S);
  printf("%d clusters\n",S->NumClusters);

  if (OutputType == 'p')
  {
    // Write the output in PDB format
    printf("Writing output PDB file...\n");
    WritePdbSolution(S,OutputFile);
  }
  else
  {
    // Write the output in a verbose log format
    printf("Writing log file...\n");
    WriteLog(S,OutputFile);
  }

  // Destroy the data structures representing the solution
  destroySolution(&S);

  return EXIT_SUCCESS;
}


// Read the command line to get the parameters:
// PrefixFile is...

void ReadCommandLine (int argc, char *argv[], char *InputFile, char *pOutputType, char *OutputFile)
{
  if (argc != 4)
  {
    printf("The command line has a wrong format!\n");
    printf("Use: %s [input_file] [output_format] [output_file]\n",argv[0]);
    printf("[output_format] is either -pdb or -log\n",argv[0]);
    exit(EXIT_WRONGCOMMANDLINE);
  }

  strcpy(InputFile,argv[1]);
  if (strcmp(argv[2],"-pdb") == 0)
    *pOutputType = 'p';
  else if (strcmp(argv[2],"-log") == 0)
    *pOutputType = 'l';
  else
  {
    printf("The command line has a wrong format!\n");
    printf("Use: %s [input_file] [output_format] [output_file]\n",argv[0]);
    printf("[output_format] is either -pdb or -log\n",argv[0]);
    exit(EXIT_WRONGCOMMANDLINE);
  }
  strcpy(OutputFile,argv[3]);

  printf("\n- PARAMETERS  -\n");
  printf("   input file    : %s \n", InputFile);
  printf("   output format : %s \n", argv[2]);
  printf("   output file   : %s \n", OutputFile);
  printf("\n");
}


// Read from the configuration file "configurazione.txt" the values of ALPHA, DELTA, ...
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA)
{
  FILE *fConfigFile;
  char Parametro[100];
  double Valore;


  fConfigFile = fopen("configurazione.txt","r");
  if (fConfigFile == NULL)
  {
    printf("Configurazione File Incorrect!\n");
    exit(EXIT_OPENFILE);
  }

  while(!feof(fConfigFile))
  {
    fscanf(fConfigFile,"%s %lf\n", Parametro, &Valore);

    if (Parametro[0] == '#')
      continue;
    else
    {
      if (!strcmp(Parametro,"ALPHA")) *ALPHA = Valore;
      if (!strcmp(Parametro,"DELTA")) *DELTA = Valore;
      if (!strcmp(Parametro,"DSHELL")) *DELTA_SHELL = Valore;
      if (!strcmp(Parametro,"DSHELL2")) *DELTA_SHELL2 = Valore;
      if (!strcmp(Parametro,"SOGLIA")) *SOGLIA = (int) Valore;

    }
  }

  fclose(fConfigFile);
}
