/// \file snaptree.c
/// \brief Main file for the computation of the tree representing the similarity
/// in a given sequence of snapshots

#include "defs.h"
#include "solution_list.h"


void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile);

//Legge il file di configurazione
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA); 

int *BuildSnapshotTree (SolutionList *SL, int NumberOfSnaps);

void SaveSnapshotTree (int NumberOfSnaps, int *NearestSnapshot, char *OutputFile);

//int CheckSnapshot(HydBondList *CurVHBL, HydBondList *OldVHBL, HydBondList *TempVHBL, int SaveFlag);

//void ViewSnapshotTree (Snapshot *RootSnap);

//void FillSnapshot (HydBondList *VHBL, Solution *S);

HydBondList *FindNewBonds (Solution *S1, Solution *S2, int *pDistance);


int main(int argc, char *argv[])
{
  char Prefix[LUNGHEZZA];	    // Name of the PDB input file (without the extension)
  unsigned int ti;            // Starting time of the sequence of snapshots
  unsigned int tf;            // Ending time of the sequence of snapshots
  char InputFile[LUNGHEZZA];	// Name of the PDB input file
  char OutputFile[LUNGHEZZA];	// Name of the output file reporting the statistics on 1st shell residual connectivity

  double ALPHA, DELTA, DELTA_SHELL, DELTA_SHELL2;
  int SOGLIA;
  unsigned int t;
  int *NearestSnapshot;


  SolutionList *SL;
  Solution *S;


  // Read the command line
  ReadCommandLine(argc,argv,Prefix,&ti,&tf,OutputFile);

  // Read the configuration file
  ReadConfigurationFile(&ALPHA, &DELTA, &DELTA_SHELL, &DELTA_SHELL2, &SOGLIA);
  
  // Create the solution list and fill it
  SL = createSolutionList();
  for (t = ti; t <= tf; t++)
  {
    S = createSolution();

    // Build the name of the input file (Prefix.[t])
    sprintf(InputFile,"%s.%u",Prefix,t);

    // Load the atoms from InputFile into solution S
    printf("Loading the atom positions from %s...",InputFile);
    LoadAtoms(InputFile,S);
    printf("%d atoms\n",S->NumAtoms);

    // Determine the hydrogen bonds in the solution
    printf("Creating the hydrogen bonds for solution %s...\n",InputFile);
    DetermineHydBonds(S,ALPHA,DELTA,DELTA_SHELL,DELTA_SHELL2);
    printf("%d bonds\n",S->NumHydBonds);

    S->Id = t;
    appendSolutionList(S,SL);
  }

  // Build the snapshot tree
  NearestSnapshot = BuildSnapshotTree(SL,tf-ti+1);

  // Save the solution
  SaveSnapshotTree(tf-ti+1,NearestSnapshot,OutputFile);

  // Destroy the solution list
  destroySolutionList(&SL);

  return EXIT_SUCCESS;
}


// Read the command line to get the parameters:
// PrefixFile is...

void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile)
{
  if (argc != 5)
  {
    printf("The command line has a wrong format!\n");
   	printf("Use: %s [input_file_prefix] [start_suffix] [end_suffix] [output_file]\n",argv[0]);
    exit(EXIT_WRONGCOMMANDLINE);
  }

  strcpy(Prefix,argv[1]);
  *pti = atoi(argv[2]);
  *ptf = atoi(argv[3]);
  strcpy(OutputFile,argv[4]);
  
  printf("\n- PARAMETERS  -\n");
  printf("   snapshot prefix  : %s \n", Prefix);
  printf("   first snapshot   : %d \n", *pti);
  printf("   last snapshot    : %d \n", *ptf);
  printf("   output file name : %s \n\n", OutputFile);
}


// Read from the configuration file "configurazione.txt" the values of ALPHA, DELTA, ...
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA)
{
  FILE *fConfigFile;
  char Parametro[LUNGHEZZA];
  double Valore;
    

  fConfigFile = fopen("configurazione.txt","r");
  if (fConfigFile == NULL)
  {
    printf("Configurazione File Incorrect!\n");
    exit(EXIT_OPENFILE);
  }
    
  while(!feof(fConfigFile))
  {
    fscanf(fConfigFile,"%s %lf\n", Parametro, &Valore);
        
    if (Parametro[0] == '#')
      continue;  
    else
    {
      if (!strcmp(Parametro,"ALPHA")) *ALPHA = Valore;
      if (!strcmp(Parametro,"DELTA")) *DELTA = Valore;
      if (!strcmp(Parametro,"DSHELL")) *DELTA_SHELL = Valore;
      if (!strcmp(Parametro,"DSHELL2")) *DELTA_SHELL2 = Valore;
      if (!strcmp(Parametro,"SOGLIA")) *SOGLIA = (int) Valore;
 
    }    
  }    

  fclose(fConfigFile);
} 


// Construct snapshot tree
int *BuildSnapshotTree (SolutionList *SL, int NumberOfSnaps)
{
  int i, j;
  int **DistanceMatrix;
  HydBondList ***NewBonds; // This matrix contains in element (i,j) a list of all hydrogen bonds appearing in snapshot j, and not in snapshot i
  // The snapshots appearing in i and not in j is contained in element (j,i). The distance matrix contains the number of elements of each list
  // The snapshot tree is the minimum spanning tree with respect to the simmetric distance d(i,j)+d(j,i)
  int *IsInTree, *NearestSnapshot, *MinDist;
  int NumInTree;
  int tMin, cMin;
  Solution *S1, *S2;//, *MF;


  DistanceMatrix = (int **) calloc(NumberOfSnaps,sizeof(int *));
  if (DistanceMatrix == NULL)
  {
    printf("Not enough memory to allocate the distance matrix!\n");
    exit(EXIT_MEMORY);
  }
  for (i = 0; i < NumberOfSnaps; i++)
  {
    DistanceMatrix[i] = (int *) calloc(NumberOfSnaps,sizeof(int));
    if (DistanceMatrix[i] == NULL)
    {
      printf("Not enough memory to allocate row %d of the distance matrix!\n",i);
      exit(EXIT_MEMORY);
    }
  }

  NewBonds = (HydBondList ***) calloc(NumberOfSnaps,sizeof(HydBondList **));
  if (NewBonds == NULL)
  {
    printf("Not enough memory to allocate the new bonds matrix!\n");
    exit(EXIT_MEMORY);
  }
  for (i = 0; i < NumberOfSnaps; i++)
  {
    NewBonds[i] = (HydBondList **) calloc(NumberOfSnaps,sizeof(HydBondList *));
    if (NewBonds[i] == NULL)
    {
      printf("Not enough memory to allocate row %d of the new bonds matrix!\n",i);
      exit(EXIT_MEMORY);
    }
  }


  IsInTree = (int *) calloc(NumberOfSnaps,sizeof(int));
  if (IsInTree == NULL)
  {
    printf("Not enough memory to allocate the incidence vector of visited snapshots!\n");
    exit(EXIT_MEMORY);
  }

  NearestSnapshot = (int *) calloc(NumberOfSnaps,sizeof(int));
  if (NearestSnapshot == NULL)
  {
    printf("Not enough memory to allocate the Nearest Snapshot vector!\n");
    exit(EXIT_MEMORY);
  }

  MinDist = (int *) calloc(NumberOfSnaps,sizeof(int));
  if (MinDist == NULL)
  {
    printf("Not enough memory to allocate the MinDist vector!\n");
    exit(EXIT_MEMORY);
  }

  // Istanzio la lista contenente tutti gli snapshot
  //AllSnaps = createSnapshotList();

  // Fill the matrices DistanceMatrix and NewBonds with the differences between the snapshots
  printf("Evaluating the distance matrix...\n");
  for (S1 = firstSolutionList(SL), i = 0; !endSolutionList(S1, SL); S1 = nextSolutionList(S1), i++)
    for (S2 = firstSolutionList(SL), j = 0; !endSolutionList(S2, SL); S2 = nextSolutionList(S2), j++)
      //if (S1 != S2)
      {
        NewBonds[i][j] = FindNewBonds(S1,S2,&DistanceMatrix[i][j]);
        printf(".");
      }


/*
  //Stampo la matrice contenente tutte le fitness
  for (i = 0; i < NumberOfSnaps; i++)
  {
    for (j = 0; j < NumberOfSnaps; j++)
    {
      printf("%4d ", DistanceMatrix[i][j]);
    }
    printf("\n");
  }

  printf("\n");
*/
  // APPLICO L'ALGORITMO DI PRIM PER LA CREAZIONE DELL'ALBERO OTTIMO


  //Inizializzo la creazione dell'albero
  NumInTree = 1;
  IsInTree[0] = TRUE;
  for (i = 0; i < NumberOfSnaps; i++)
  {
    NearestSnapshot[i] = 0;
    MinDist[i] = DistanceMatrix[i][0] + DistanceMatrix[0][i];
  }

  while (NumInTree < NumberOfSnaps)
  {
    // Prelevo lo snapshot pi "vicino" a quelli presenti correntemente nell'albero
    tMin = -1;
    cMin = firstSolutionList(SL)->NumResiduals * firstSolutionList(SL)->NumResiduals;
    for (j = 1; j < NumberOfSnaps; j++)
      if ( (!IsInTree[j]) && (MinDist[j] < cMin) )
      {
        cMin = MinDist[j];
        tMin = j;
      }

    // Inserisco lo snapshot all'interno dell'albero
    IsInTree[tMin] = TRUE;
    NumInTree++;

    // Aggiorno le distanze minime per gli altri snapshot
    for (j = 1; j < NumberOfSnaps; j++)
      if ( (!IsInTree[j]) && (DistanceMatrix[j][tMin] + DistanceMatrix[tMin][j] < MinDist[j]) )
      {
        MinDist[j] = DistanceMatrix[j][tMin] + DistanceMatrix[tMin][j];
        NearestSnapshot[j] = tMin;
      }
  }


  for (i = 0; i < NumberOfSnaps; i++)
    free(DistanceMatrix[i]);
  free(DistanceMatrix);

  for (i = 0; i < NumberOfSnaps; i++)
    free(NewBonds[i]);
  free(NewBonds);

  free(IsInTree);
  free(MinDist);

  return NearestSnapshot;
}


HydBondList *FindNewBonds (Solution *S1, Solution *S2, int *pDistance)
{
  ResidualPos *R1, *R2;
  HydBond *HB1 , *HB2, *HBnew;
  HydBondList *HBL;


  HBL = createHydBondList();

  *pDistance = 0;

  // Find all hydrogen bonds which appear in solution S2 and not in solution S1
  for (R2 = firstResidualList(S2->RL); !endResidualList(R2,S2->RL); R2 = nextResidualList(R2))
    for (HB2 = firstHydBondList(R2->BL); !endHydBondList(HB2,R2->BL); HB2 = nextHydBondList(HB2))
    {
      // Find residual R2 in solution S1 (it must exist!)
      for (R1 = firstResidualList(S1->RL); (R1->Id != R2->Id) && !endResidualList(R1,S1->RL); R1 = nextResidualList(R1));

      if (R1->Id != R2->Id) 
      {
        printf("Solutions %d and %d ");
        exit(EXIT_INCONSISTENCY);
      }

      // Find whether a hydrogen bond between R1 and R2 exists in solution S1
      // ATTENZIONE: COSI' IL LEGAME DEVE ESSERE LO STESSO (CIOE' FRA GLI STESSI ATOMI)!!!!
      //             COMMENTATO E' IL CODICE PER CONTARE LEGAMI GENERICI (PURCHE' FRA I DUE RESIDUI)
      for (HB1 = firstHydBondList(R1->BL); !endHydBondList(HB1,R1->BL); HB1 = nextHydBondList(HB1))
        //if ( ( (HB1->R1->Id == HB2->R1->Id) && (HB1->R2->Id == HB2->R2->Id) ) ||
        //     ( (HB1->R1->Id == HB2->R2->Id) && (HB1->R2->Id == HB2->R1->Id) ) )
        if ( (HB1->A->Id == HB2->A->Id) && (HB1->H->Id == HB2->H->Id) && (HB1->D->Id == HB2->D->Id) ) break;

      if (endHydBondList(HB1,R1->BL))
      {
        (*pDistance)++;
        HBnew = createHydBond(HB2->Id,HB2->Distance,HB2->Angle,HB2->A,HB2->D,HB2->H,HB2->DonRes,HB2->AccRes);
        appendHydBondList(HBnew,HBL);
      }
    }

  return HBL;
}


void SaveSnapshotTree (int NumberOfSnaps, int *NearestSnapshot, char *OutputFile)
{
  FILE *fOutputFile;
  int t;


  fOutputFile = fopen(OutputFile, "w");
  if (fOutputFile == NULL)
  {
    printf("File %s could not be opened!\n",OutputFile);
    exit(EXIT_OPENFILE);
  }

  for (t = 0; t < NumberOfSnaps; t++)
    if (t != NearestSnapshot[t])
      fprintf(fOutputFile,"%d %d %d\n",t,NearestSnapshot[t],0);

  fclose(fOutputFile);
}


/*
//Check two snapshot and return the fitness (number of variations)
int CheckSnapshot(HydBondList *CurVHBL, HydBondList *OldVHBL, HydBondList *TempVHBL, int SaveFlag)
{
  int Fitness = 0;
  int CurBond = 0;
  int A;
  int i, K, j, RMax, CheckFlag;
  VarHydBond *VH1, *VH2, *TempVH;



  // Vengono cercati tutti i legami che sono nella soluzione nuova e non in quelle precedenti
  for(VH1 = firstHydBondList(CurVHBL); !endHydBondList(VH1, CurVHBL); VH1 = nextHydBondList(VH1))
  {
    CheckFlag = 0;
    for(VH2 = firstHydBondList(OldVHBL); !endHydBondList(VH2, OldVHBL); VH2 = nextHydBondList(VH2))
    {
      if(((VH1->R1 == VH2->R1) && (VH1->R2 == VH2->R2)) || ((VH1->R1 == VH2->R2) && (VH1->R2 == VH2->R1)))
      {
        Fitness++;
        CheckFlag = 1;
      }
    }


    if(CheckFlag == 0 && SaveFlag == 1)
    {
      TempVH = (VarHydBond *) createVarHydBond(VH1->R1, VH1->R2, 1);
      appendHydBondList(TempVH, TempVHBL);
    }
  }


  // Vengono cercati tutti i legami che erano presenti nelle soluzioni precedenti e non in quella nuova
  for(VH1 = firstHydBondList(OldVHBL); !endHydBondList(VH1, OldVHBL); VH1 = nextHydBondList(VH1))
  {
    CheckFlag = 0;
    for(VH2 = firstHydBondList(CurVHBL); !endHydBondList(VH2, CurVHBL); VH2 = nextHydBondList(VH2))
    {
      if(((VH1->R1 == VH2->R1) && (VH1->R2 == VH2->R2)) || ((VH1->R1 == VH2->R2) && (VH1->R2 == VH2->R1)))
      {
        Fitness++;
        CheckFlag = 1;
      }
    }


    if(CheckFlag == 0 && SaveFlag == 1)
    {
      TempVH = (VarHydBond *) createVarHydBond(VH1->R1, VH1->R2, -1);
      appendHydBondList(TempVH, TempVHBL);
    }
  }

  return Fitness;
}


// Visualizza l'albero degli snapshot che  stato generato
void ViewSnapshotTree(Snapshot *RootSnap)
{
  Snapshot *CurSnap;

  if(RootSnap->Id != 1)
  {
    printf("%d - %d\n", RootSnap->Id, RootSnap->parent->Id);
  }

  for(CurSnap = (Snapshot *) firstSnapshotList(RootSnap->childs); !endSnapshotList(CurSnap, RootSnap->childs); CurSnap = (Snapshot *) nextSnapshotList(CurSnap))
  {
    ViewSnapshotTree(CurSnap);
  }
}


// Inserisce all'interno di uno snapshot tutti i legami presenti nella soluzione corrispondente
void FillSnapshot (HydBondList *VHBL, Solution *S)
{
  Residual *R1;
  HydBond *H1;
  VarHydBond *VH1;

  for(R1 = firstResidualList(S->RL); !endResidualList(R1, S->RL); R1 = nextResidualList(R1))
  {
    for(H1 = firstHydBondList(R1->BL); !endHydBondList(H1, R1->BL); H1 = nextHydBondList(H1))
    {
      VH1 = (VarHydBond *) createVarHydBond(H1->R1->Id, H1->R2->Id, 1);
      appendHydBondList(VH1, VHBL);
    }
  }
}

*/
