/// \file residual.h
/// \brief Definition of data structures and functions to handle residuals.
///    

#ifndef _residual_h
#define _residual_h


typedef struct _residual_s Residual;
typedef Residual ResidualList;
typedef Residual ResidualPos;

#include "atom_list.h"
#include "hydbond_list.h"
#include "cluster.h"

/// \brief Data structure representing a residual
///
/// A residual has a unique numerical index, a string representing the chemical substance,
/// three lists of atoms (potential acceptors or donors, hydrogen atoms, other atoms),
/// a number and a list of hydrogen bonds involving the residual and a pointer to the
/// cluster including it.
/// It also has pointers to the next and the previous residual in the list it belongs to.

struct _residual_s
{
    int Id; ///< Numerical index of the residual
    char Substance[4]; ///< Chemical substance to which the residual belongs

    AtomList *ADL; ///< List of the donor/acceptor atoms belonging to the residual
    AtomList *HL;  ///< List of the hydrogen atoms belonging to the residual
    AtomList *OL;  ///< List of the other atoms belonging to the residual  

    int NumHydBonds; ///< Number of hydrogen bonds to which the residual takes part
    HydBondList *BL; ///< List of the hydrogen bonds to which the residual takes part
    Cluster  *C;     ///< Cluster to which the residual belongs (NULL if none)

    Residual *next; ///< Pointer to the next     element in the list to which the residual belongs (NULL if none)
    Residual *prev; ///< Pointer to the previous element in the list to which the residual belongs (NULL if none)
};

/// Create a new residual
Residual *createResidual (int Id, char *Substance);

/// The residual is emptied (all fields zeroed), but not destroyed!
void emptyResidual (Residual *R);

#endif
