/// \file hb-fs.c
/// \brief Main file for the computation of the number of residuals in first shell
/// in a given sequence of snapshots.

#include "defs.h"
#include "solution.h"


void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile);

//Legge il file di configurazione
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA);

int FindWaterFirstShellBondSolute (Solution *S, ResidualList *WL, ResidualList *SL);

int FindWaterFirstShellBondWater (Solution *S, ResidualList *WL, double DELTA_SHELL2);


int main(int argc, char *argv[])
{
  char Prefix[LUNGHEZZA];     // Name of the PDB input file (without the extension)
  unsigned int ti;            // Starting time of the sequence of snapshots
  unsigned int tf;            // Ending time of the sequence of snapshots
  char InputFile[LUNGHEZZA];  // Name of the PDB input file
  char OutputFile[LUNGHEZZA]; // Name of the output file reporting the statistics on 1st shell residual connectivity

  double ALPHA, DELTA, DELTA_SHELL, DELTA_SHELL2;
  int SOGLIA, NFS;
  unsigned int t;

  Solution *S;
  ResidualList *FirstShellWaterList, *SoluteList;

  FILE *Results;


  // Read the command line
  ReadCommandLine(argc,argv,Prefix,&ti,&tf,OutputFile);

  // Read the configuration file
  ReadConfigurationFile(&ALPHA, &DELTA, &DELTA_SHELL, &DELTA_SHELL2, &SOGLIA);

  // Build the name of the log file (nomefileinput.log.txt)
  //sprintf(LogFile,"%s.log.txt",InputFile);

  // Inizializzo il file contenente i risultati delle elaborazioni
  Results = fopen(OutputFile, "w");
  if (Results == NULL)
  {
    printf("File hb-fs.txt could not be opened!\n");
    exit(EXIT_OPENFILE);
  }

  for(t=ti; t<=tf; t++)
  {
    // Build the name of the starting input file (nomefileinput.1)
    sprintf(InputFile,"%s.%u",Prefix,t);

    // Create an empty solution
    S = createSolution();
    FirstShellWaterList = createResidualList();
    SoluteList = createResidualList();

    // Load the atoms from InputFile into solution S
    printf("Loading the atom positions from %s...",InputFile);
    LoadAtoms(InputFile,S);
    printf("%d atoms\n",S->NumAtoms);

    // Determine the hydrogen bonds in the solution
    printf("Creating the hydrogen bonds for solution %s...\n",InputFile);
    DetermineHydBonds(S,ALPHA,DELTA,DELTA_SHELL,DELTA_SHELL2);
    printf("%d bonds\n", S->NumHydBonds);


    fprintf(Results, "Snapshot %d\n\n", t);

    // Search solute residuals
    NFS = FindSoluteResiduals(S, SoluteList);

    // Search residual in 1 shell
    printf("Searching 1^ shell water residuals...\n");
    NFS = FindFirstShellWaterResidual (S, FirstShellWaterList, DELTA_SHELL);
    fprintf(Results, "Water residuals in 1st shell: %d\n", NFS);

    // Search residual in 1 shell with solute bonds
    printf("Searching 1^ shell water residuals bonded with solute...\n");
    NFS = FindWaterFirstShellBondSolute (S, FirstShellWaterList, SoluteList);
    fprintf(Results, "Water residuals in 1st shell bonded with solute: %d\n", NFS);

    // Search residual in 1 shell with solute bonds
    printf("Searching 1^ shell water residuals bonded with water...\n");
    NFS = FindWaterFirstShellBondWater (S, FirstShellWaterList, DELTA_SHELL2);
    fprintf(Results, "Water residuals in 1st shell bonded with water (also in 2nd shell): %d\n", NFS);
    fprintf(Results, "\n\n\n");
  }

  fclose(Results);

  // Destroy the data structures representing the solution
  destroySolution(&S);

  return EXIT_SUCCESS;
}


// Read the command line to get the parameters:
// PrefixFile is...

void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile)
{
  if (argc != 5)
  {
    printf("The command line has a wrong format!\n");
 printf("Use: %s [input_file_prefix] [start_suffix] [end_suffix] [output_file]\n",argv[0]);
    exit(EXIT_WRONGCOMMANDLINE);
  }

  strcpy(Prefix,argv[1]);
  *pti = atoi(argv[2]);
  *ptf = atoi(argv[3]);
  strcpy(OutputFile,argv[4]);

  printf("\n- PARAMETERS  -\n");
  printf("   snapshots prefix : %s \n", Prefix);
  printf("   first snapshot   : %d \n", *pti);
  printf("   last snapshot    : %d \n", *ptf);
  printf("   output file name : %s \n\n", OutputFile);
}


// Read from the configuration file "configurazione.txt" the values of ALPHA, DELTA, ...
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA)
{
  FILE *fConfigFile;
  char Parametro[LUNGHEZZA];
  double Valore;


  fConfigFile = fopen("configurazione.txt","r");
  if (fConfigFile == NULL)
  {
    printf("Configurazione File Incorrect!\n");
    exit(EXIT_OPENFILE);
  }

  while(!feof(fConfigFile))
  {
    fscanf(fConfigFile,"%s %lf\n", Parametro, &Valore);

    if (Parametro[0] == '#')
      continue;
    else
    {
      if (!strcmp(Parametro,"ALPHA")) *ALPHA = Valore;
      if (!strcmp(Parametro,"DELTA")) *DELTA = Valore;
      if (!strcmp(Parametro,"DSHELL")) *DELTA_SHELL = Valore;
      if (!strcmp(Parametro,"DSHELL2")) *DELTA_SHELL2 = Valore;
      if (!strcmp(Parametro,"SOGLIA")) *SOGLIA = (int) Valore;

    }
  }

  fclose(fConfigFile);
}


// Find water residual in 1 shell with solute bonds
int FindWaterFirstShellBondSolute (Solution *S, ResidualList *WL, ResidualList *SL)
{
  int NumberWaterBondSolute = 0;
  Residual *R1, *R2;
  HydBond *H1;

  for(R1 = firstResidualList(SL); !endResidualList(R1, SL); R1 = nextResidualList(R1))
  {
    for(R2 = firstResidualList(WL); !endResidualList(R2, WL); R2 = nextResidualList(R2))
    {
      for(H1 = firstHydBondList(R1->BL); !endHydBondList(H1, R1->BL); H1 = nextHydBondList(H1))
      {
        if((H1->R1->Id == R2->Id) || (H1->R2->Id == R2->Id))
        {
          NumberWaterBondSolute++;
        }
      }
    }
  }

  return NumberWaterBondSolute;
}


// Find water residual in 1 shell with water bonds
int FindWaterFirstShellBondWater (Solution *S, ResidualList *WL, double DELTA_SHELL2)
{
  int NumberWaterBondWater = 0;
  int FoundFlag = 0;
  ResidualList *RSS;
  Residual *R1, *R2;
  HydBond *H1;

  RSS = (ResidualList *) createResidualList();


  // Preleva tutte le molecole d'acqua presenti entro la seconda shell
  for(R2 = firstResidualList(S->RL); !endResidualList(R2, S->RL); R2 = nextResidualList(R2))
  {
    if(strcmp(R2->Substance, "WAT") == 0)
    {
      if (BelongsToShell(S, R2, DELTA_SHELL2) == TRUE)
      {
        R1 = createResidual(R2->Id, R2->Substance);
        R1->NumHydBonds = R2->NumHydBonds;
        R1->BL = R2->BL;
        R1->ADL = R2->ADL;
        R1->HL = R2->HL;
        R1->OL = R2->OL;
        R1->next = R2->next;
        R1->prev = R2->prev;
        R1->C = R2->C;
        appendResidualList(R1, RSS);
      }
    }
  }


  // Per ciascuna molecola presente nella prima shell ne cerca i legami con una presente
  // nella prima o nella seconda shell
  for (R1 = firstResidualList(WL); !endResidualList(R1, WL); R1 = nextResidualList(R1))
  {
    FoundFlag = 0;
    for(R2 = firstResidualList(RSS); !endResidualList(R2, RSS); R2 = nextResidualList(R2))
    {
      for(H1 = firstHydBondList(R1->BL); !endHydBondList(H1, R1->BL); H1 = nextHydBondList(H1))
      {
        if(R1->Id != R2->Id)
        {
          if((H1->R1->Id == R2->Id) || (H1->R2->Id == R2->Id))
          {
            if((strcmp(H1->R1->Substance, "WAT") == 0) && (strcmp(H1->R2->Substance, "WAT") == 0))
            {
              FoundFlag = 1;
            }
          }
        }
      }
    }

    if(FoundFlag == 1) NumberWaterBondWater++;
  }

  return NumberWaterBondWater;
}
