/// \file hb-degree.c
/// \brief Main file for the computation of the degree of the water residuals
/// in first shell in a given sequence of snapshots.

#include "defs.h"
#include "solution.h"


// Parse the command line
void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile);

// Read the configuration file
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA);

int *WaterBondsFirstShell (Solution *S, ResidualList *RL);


int main(int argc, char *argv[])
{
  char Prefix[LUNGHEZZA];     // Name of the PDB input file (without the extension)
  unsigned int ti;            // Starting time of the sequence of snapshots
  unsigned int tf;            // Ending time of the sequence of snapshots
  char InputFile[LUNGHEZZA]; // Name of the PDB input file
  char OutputFile[LUNGHEZZA]; // Name of the output file reporting the statistics on 1st shell residual connectivity

  double ALPHA, DELTA, DELTA_SHELL, DELTA_SHELL2;
  unsigned int t, k;
  int SOGLIA, NFS;

  Solution *S;
  ResidualList *FirstShellResidual, *SoluteResidual;
  int *WaterBondStatistics;

  FILE *Results;


  // Read the command line
  ReadCommandLine(argc,argv,Prefix,&ti,&tf,OutputFile);

  // Read the configuration file
  ReadConfigurationFile(&ALPHA, &DELTA, &DELTA_SHELL, &DELTA_SHELL2, &SOGLIA);

  // Inizializzo il file contenente i risultati delle elaborazioni
  Results = fopen(OutputFile, "w");
  if (Results == NULL)
  {
    printf("File hb-degree.txt could not be opened!\n");
    exit(EXIT_OPENFILE);
  }

  for (t = ti; t <= tf; t++)
  {
    // Build the name of the current input file (PrefixFile.Number)
    sprintf(InputFile,"%s.%u",Prefix,t);

    // Create an empty solution
    S = createSolution();

    // Load the atoms from InputFile into solution S
    printf("Loading the atom positions from %s...",InputFile);
    LoadAtoms(InputFile,S);
    printf("%d atoms\n",S->NumAtoms);

    // Determine the hydrogen bonds in the solution
    printf("Creating the hydrogen bonds for solution %s...\n",InputFile);
    DetermineHydBonds(S,ALPHA,DELTA,DELTA_SHELL,DELTA_SHELL2);

    // Search solute residuals
    printf("Searching 1^ shell residual...\n");
    SoluteResidual = createResidualList();
    NFS = FindSoluteResiduals(S, SoluteResidual);
    fprintf(Results, "%d solute residuals found\n\n", NFS);

    // Search water residuals in 1 shell
    FirstShellResidual = createResidualList();
    NFS = FindFirstShellWaterResidual (S, FirstShellResidual, DELTA_SHELL);
    fprintf(Results, "%d water residuals found in the 1st shell\n\n", NFS);

    // Calculating water bond statistics for first shell residual
    WaterBondStatistics = WaterBondsFirstShell(S, FirstShellResidual);
    for(k = 0; k <= 5; k++)
      fprintf(Results, "Residual with %d bonds: %d\n", k, WaterBondStatistics[k]);
    fprintf(Results, "\n\n");

    // Destroy the data structures representing the solution
    destroySolution(&S);
  }

  fclose(Results);

  return EXIT_SUCCESS;
}


// Read the command line to get the parameters:
// PrefixFile is...

void ReadCommandLine (int argc, char *argv[], char *Prefix, int *pti, int *ptf, char *OutputFile)
{
  if (argc != 5)
  {
    printf("The command line has a wrong format!\n");
    printf("Use: %s [input_file_prefix] [start_suffix] [end_suffix] [output_file]\n",argv[0]);
    exit(EXIT_WRONGCOMMANDLINE);
  }

  strcpy(Prefix,argv[1]);
  *pti = atoi(argv[2]);
  *ptf = atoi(argv[3]);
  strcpy(OutputFile,argv[4]);

  printf("\n- PARAMETERS  -\n");
  printf("   snapshots prefix : %s \n", Prefix);
  printf("   first snapshot   : %d \n", *pti);
  printf("   last snapshot    : %d \n", *ptf);
  printf("   output file name : %s \n\n", OutputFile);
}


// Read from the configuration file "configurazione.txt" the values of ALPHA, DELTA, ...
void ReadConfigurationFile (double *ALPHA, double *DELTA, double *DELTA_SHELL, double *DELTA_SHELL2, int *SOGLIA)
{
  FILE *fConfigFile;
  char Parametro[100];
  double Valore;


  fConfigFile = fopen("configurazione.txt","r");
  if (fConfigFile == NULL)
  {
    printf("Configurazione File Incorrect!\n");
    exit(EXIT_OPENFILE);
  }

  while(!feof(fConfigFile))
  {
    fscanf(fConfigFile,"%s %lf\n", Parametro, &Valore);

    if (Parametro[0] == '#')
      continue;
    else
    {
      if (!strcmp(Parametro,"ALPHA")) *ALPHA = Valore;
      if (!strcmp(Parametro,"DELTA")) *DELTA = Valore;
      if (!strcmp(Parametro,"DSHELL")) *DELTA_SHELL = Valore;
      if (!strcmp(Parametro,"DSHELL2")) *DELTA_SHELL2 = Valore;
      if (!strcmp(Parametro,"SOGLIA")) *SOGLIA = (int) Valore;

    }
  }

  fclose(fConfigFile);
}


// Count the number of bonds with water residuals for every first shell residual
int *WaterBondsFirstShell (Solution *S, ResidualList *RL)
{
  int *WaterBondStatistics;
  //int BondCount;
  Residual *R1;
  //HydBond *HB;

  WaterBondStatistics = (int *) calloc(6, sizeof(int));
/*
  // NOTA BENE: Codice valido se si vogliono contare solo i legami fra acqua e acqua
  for(R1 = firstResidualList(RL); !endResidualList(R1, RL); R1 = nextResidualList(R1))
  {
    BondCount = 0;
    for (HB = firstHydBondList(R1->BL); !endHydBondList(HB, R1->BL); HB = nextHydBondList(HB))
    {
      if ( (HB->R1->Id != R1->Id) && (strcmp(HB->R1->Substance, "WAT") == 0) ) BondCount++;
      if ( (HB->R2->Id != R1->Id) && (strcmp(HB->R2->Substance, "WAT") == 0) ) BondCount++;
    }
    WaterBondStatistics[BondCount]++;
  }
*/
  // NOTA BENE: Codice valido se si vogliono contare tutti i legami
  for (R1 = firstResidualList(RL); !endResidualList(R1, RL); R1 = nextResidualList(R1))
  {
    if (R1->NumHydBonds > 5)
    {
      printf("Water residual %d has %d > 5 hydrogen bonds!",R1->Id,R1->NumHydBonds);
      exit(EXIT_INCONSISTENCY);
    }

    if (strcmp(R1->Substance,"WAT") == 0)
      WaterBondStatistics[R1->NumHydBonds]++;
  }

  return WaterBondStatistics;
}
