# Dichiaro la codifica del mio script python (UTF-8), mi semplifica alcuni passi come l'uso di caratteri accentati nei commenti
# -*- coding: utf-8 -*-
"""
Esempio di script python. Crea un server che sincronizza gli accessi ad un buffer comune attraverso messaggi.
In una prima fase il server permette solo la scrittura. Una volta riempito il buffer, il server compie alcuni calcoli (in questo caso la media) e si mette in ascolto per comunicarne il risultato.

Tutto ciò che viene scritto nei commenti con triple virgolette, può essere utilizzato da python per la documentatione dei file tramite l'utility pydoc.
"""

import socket		# importazione modulo per l'uso dei socket

class Buffer(object):
	"""
	Classe che implementa un buffer.
	Il buffer consente di inserire (push) ed estrarre (pop) elementi. Non è possible aggiungere elementi se si è raggiunta la capacità massima.
	"""
	
	def __init__(self,max=10):
		"""
		Costruttore del buffer. 
		La dimensione massima del buffer è di default 10 elementi.
		"""
		self.buf = [] 
		self.max = max
	
	def push(self,e):
		"""
		Inserisce un elemento nel buffer se non è pieno.
		Lancia un'eccezione di tipo BufferException se il buffer è pieno.
		"""
		
		if len(self.buf) >= self.max:
			raise BufferException()
		self.buf.append(e)
	
	def pop(self):
		"""
		Estrae un elemento da buffer se non è vuoto.
		Lancia un'eccezione di tipo BufferException se il buffer è pieno
		"""
		if len(self.buf) == 0:
			raise BufferException
		return self.buf.pop(0)

class BufferException(Exception):
	"""
	Eccezione per segnalare problemi col buffer.
	Non aggiunge nulla di diverso alle normali eccezioni.
	"""
	pass
		
class BufferIter(Buffer):
	"""
	Buffer con iteratore.
	Estende la classe Buffer ed aggiunge la capacità di iterare sugli elementi del buffer.
	Per creare un iteratore è necessario implementare i metodi __iter__ e __next__, che restituiscono rispettivamente l'iteratore stesso e l'elemento successivo nell'iterazione.
	"""
	def __init__(self,max):
		super(BufferIter,self).__init__(max)
		self.current = 0
		
	def __iter__(self):
		self.current = 0
		return self

	def __next__(self):
		"""
		Restituisce l'elemento sucessivo del buffer.
		Per terminare il ciclo è necessario sollevare l'eccezione StopIteration.
		"""
		if self.current >= len(self.buf):
			raise StopIteration
		else:
			self.current += 1
			return self.buf[self.current-1]

class Server(object):
	"""
	Classe che implementa il server.
	"""
	def __init__(self):
		"""
		Costruttore del server.
		Alloca e inizializza il buffer e i socket necessari per lo scambio di messaggi.
		"""
		print('Inizializzazione server')
		# Versione 1: Buffer normale
		# self.buffer = Buffer(4)
		# Versione 2: Buffer con iteratore
		self.buffer = BufferIter(4) 
		
		# Socket write creato per la scrittura nel buffer su porta 8001
		self.write = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
		self.write.bind(('localhost',8001))
		self.write.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
		# Socket read creato per la letura dei risultati su porta 8002
		self.read = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
		self.read.bind(('localhost',8002))
		self.read.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

	
	def start(self):
		"""
		Avvia il server.
		"""
		print('Avvio ascolto')
		self.write.listen(0)
		self.read.listen(0)
		
		
		# Fase 1: ascolto e ricezione per scrittura nel buffer. 
		print('Avvio ricezione dati')
		try:
			while True:
				# La accept è bloccate e permette di leggere uno alla volta i dati inviati dai processi concorrenti.
				conn, add = self.write.accept()
				data = conn.recv(100) # Ricezione di massimo 100 caratteri dallo stream
				conn.close()
				self.buffer.push(data) # Quando ricevo una quantità di dati maggiore della capienza del buffer, l'eccezione sollevata dal metodo push fa terminare il ciclo
				print('\tRicevuto {}'.format(data))
		except BufferException:
			print('Interrompo ricezione')
		finally:	
			self.write.close()
	
		# Fase 2: i dati vengono elaborati 
		print('Calcolo media')
		media = 0
		cont = 0
		# Versione 1: estraggo tutti i  dati dal buffer e ne faccio la media
		"""try: 
			while True:
				media += int(self.buffer.pop().decode('UTF-8'))
				cont += 1
		except BufferException:
			pass"""
		# Versione 2: scorro i dati attraverso l'iteratore senza estrarli dal buffer (che rimane pieno)
		for i in self.buffer:
			media += int(i.decode('UTF-8'))
			cont += 1
		media /= float(cont)
		
		#Fase 3: mi metto in ascolto per processi che vogliono il risultato della computazione
		print('Avvio invio dati')
		while True:
			conn, add = self.read.accept()
			conn.sendall(bytes(str(media),'UTF-8'))
			conn.close()	
		self.read.close()
	
	def stop(self):
		"""
		Chiude i socket aperti dal server.
		"""
		self.read.close()
		self.write.close()
		

if __name__ == '__main__':
	# Racchiudo l'esecuzione del server all'interno di un try per catturare l'interruzione da tastiera
	try:
		s = Server()
		s.start()
	except KeyboardInterrupt:
		s.stop()
		print("Server terminato")
		
		

